package ru.yandex.chemodan.app.docviewer.web.backend;

import lombok.Data;
import org.dom4j.Document;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.StoredUriManager;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.crypt.TokenManager;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequestWithUid;
import ru.yandex.chemodan.app.docviewer.web.backend.DownloadUrlAction.DownloadUrlRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author akirakozov
 */
@SuppressWarnings("serial")
@Data
public class DownloadUrlAction extends AbstractXmlActionServlet<DownloadUrlRequest> implements BackendServlet {

    public static class DownloadUrlRequest extends DocviewerRequestWithUid {

        @ActionParameter
        @Nullable
        String id;

        @ActionParameter("archive-path")
        @Nullable
        String archivePath;

        @ActionParameter
        @Nullable
        String url;

        @ActionParameter
        @Nullable
        String disposition;

        @Nullable
        @ActionParameter("serp-params")
        String serpParams = null;
    }

    @Value("${mpfs.host}")
    private String mpfsHost;

    @Autowired
    private WebSecurityManager webSecurityManager;
    @Autowired
    private StoredUriManager storedUriManager;
    @Autowired
    DiskManager diskManager;
    @Autowired
    private TokenManager tokenManager;
    @Autowired
    private UriHelper uriHelper;

    @Override
    public String getActionUrl() {
        return "/download-url";
    }

    @Override
    public void execute(DownloadUrlRequest request, Document doc) {
        if (request.uid == null) {
            throw new BadRequestException("User ID ('uid') is not specified");
        }

        ActualUri uri;
        if (StringUtils.isNotEmpty(request.id) && StringUtils.isEmpty(request.url)) {
            webSecurityManager.validateFileRightUsingUid(request.uid, request.id);
            StoredUri storedUri = storedUriManager.findByFileIdAndUidO(request.id, request.uid)
                    .getOrThrow("Could not find stored uri by file id " + request.id + " and uid " + request.uid);
            uri = storedUri.getUri();
        } else if (StringUtils.isNotEmpty(request.url) && StringUtils.isEmpty(request.id)) {
            webSecurityManager.validateUrl(request.url, request.uid, Option.ofNullable(request.serpParams));
            DocumentSourceInfo source = DocumentSourceInfo.builder().originalUrl(request.url).uid(request.uid).build()
                    .withShowNda(request.showNda);
            uri = uriHelper.rewrite(source);
        } else {
            throw new BadRequestException("File id and url cannot be both empty or both defined");
        }

        Option<String> archivePath = StringUtils.notEmptyO(request.archivePath);

        boolean isMpfsHost = mpfsHost.equals(uri.getUri().getHost());
        boolean isFileInArchive = archivePath.isPresent();

        String downloadUrl;
        if (isMpfsHost && !isFileInArchive) {
            downloadUrl = diskManager.getDiskDownloadUrl(
                    uri, isContentDispositionInline(request), request.uid);
        } else {
            Tuple2<String, String> tokenAndTs = tokenManager.getTokenAndTs(Option.ofNullable(request.id),
                    Option.ofNullable(request.url), archivePath, request.uid);

            if (StringUtils.isNotEmpty(request.id)) {
                downloadUrl = UrlUtils.addParameter("./source", "id", request.id);
            } else {
                downloadUrl = UrlUtils.addParameter("./source", "url", request.url);
            }
            if (isFileInArchive) {
                downloadUrl = UrlUtils.addParameter(downloadUrl, "archive-path", archivePath.get());
            }

            if (isContentDispositionInline(request)) {
                downloadUrl = UrlUtils.addParameter(downloadUrl, "inline", true);
            }
            // uid is added by verstka, and it will be verified via token
            downloadUrl = UrlUtils.addParameter(downloadUrl,
                    "ts", tokenAndTs.get2(), "token", tokenAndTs.get1());
        }

        doc.addElement("download-url").addText(downloadUrl);
    }

    private boolean isContentDispositionInline(DownloadUrlRequest request) {
        return Option.ofNullable(request.disposition).isSome("inline");
    }

}
