package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.List;

import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.DocumentProperties;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.Document.Page;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.PdfPageWordsExtractor;
import ru.yandex.chemodan.app.docviewer.utils.pdf.text.WordPositionSerializer;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequestWithUid;
import ru.yandex.chemodan.app.docviewer.web.backend.ExtractTextAction.ExtractTextRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author akirakozov
 */
@SuppressWarnings("serial")
public class ExtractTextAction extends AbstractXmlActionServlet<ExtractTextRequest> implements BackendServlet {

    private static final Logger logger = LoggerFactory.getLogger(ExtractTextAction.class);

    public static class ExtractTextRequest extends DocviewerRequestWithUid {
        @ActionParameter
        public String id;

        // 1-based page index
        @ActionParameter
        public int page;

        @ActionParameter
        public String query;
    }

    @Autowired
    private StoredResultHelper storedResultHelper;

    @Autowired
    private WebSecurityManager webSecurityManager;

    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;

    @Autowired
    private TemporaryFileCache temporaryFileCache;

    @Override
    public String getActionUrl() {
        return "/extract-text";
    }

    @Override
    public void execute(ExtractTextRequest request, Document doc) {
        webSecurityManager.validateFileRightUsingUid(request.uid, request.id);

        StoredResult storedResult = storedResultHelper.getValidatedResult(request.id, getConvertTargetMobileIncluded(request));
        storedResultHelper.validatePage(storedResult, request.page);

        Option<String> fileTextContentLink = storedResult.getDocumentProperties().getO(DocumentProperties.EXTRACTED_TEXT);

        Element e = doc.addElement("extract-text");
        if (fileTextContentLink.isPresent()) {
            logger.info("Get text content from cache: {}", fileTextContentLink);
            resultHolder.withTempFile(resolve(fileTextContentLink),
                    resultFile -> e.add(getTextPositionsFromContent(request.page, resultFile)));
        } else {
            resultHolder.withTempFile(resolve(storedResult.getFileLink()),
                    resultFile -> e.add(getTextPositions(request.page, resultFile)));
        }
    }

    private FileLink resolve(Option<String> fileLink) {
        return resultHolder.toFileLink(fileLink.get());
    }

    private Element getTextPositionsFromContent(int page, File2 resultFile) {
        ListF<Page> pages = Cf.x(WordPositionSerializer.deserializeJsonCompressed(resultFile.readBytes()).getPages());
        Option<Page> p = pages.getO(page - 1);
        return wrap(p, page);
    }

    private Element wrap(Option<Page> p, int page) {
        Page rp = p.getOrElse(generateEmptyPage(page));
        return Dom4jUtils.readRootElement(WordPositionSerializer.serializeToBytes(
                new ru.yandex.chemodan.app.docviewer.utils.pdf.text.Document(Option.of(rp))));
    }

    private Element wrap(List<Page> p, int page) {
        return wrap(p.isEmpty() ? Option.empty() : Option.ofNullable(p.get(0)), page);
    }

    private Page generateEmptyPage(int page) {
        return new Page(page, 0, 0, 0, 0, Cf.list());
    }

    private Element getTextPositions(int page, File2 resultFile) {
        return PdfUtils.withExistingDocument(resultFile, true, a -> {
            return wrap(PdfPageWordsExtractor.getDocumentWithExtractedWords(a, page - 1, page - 1).getPages(), page);
        });
    }

}
