package ru.yandex.chemodan.app.docviewer.web.backend;

import org.dom4j.Document;
import org.dom4j.Element;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsUtils;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequest;
import ru.yandex.chemodan.app.docviewer.web.backend.GetOfficeRedactorUrlAction.OfficeActionCheckRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.dataSize.DataSize;

/**
 * @author akirakozov
 */
public class GetOfficeRedactorUrlAction
        extends AbstractXmlActionServlet<OfficeActionCheckRequest> implements BackendServlet
{
    private static final String NOT_SUPPORTED = "not-supported";

    public static class OfficeActionCheckRequest extends DocviewerRequest {
        @ActionParameter(required = true)
        PassportUidOrZero uid;

        @ActionParameter
        String name;

        @ActionParameter(required = true)
        String url;

        @ActionParameter
        @Nullable
        String yandexuid;
    }

    @Autowired
    private WebSecurityManager webSecurityManager;
    @Autowired
    private DiskManager diskManager;
    @Autowired
    private StoredUriDao storedUriDao;

    @Override
    public String getActionUrl() {
        return "/get-office-redactor-url";
    }

    @Override
    public void execute(OfficeActionCheckRequest request, Document doc) {
        if (request.uid.isAuthenticated() && request.uid.isYandexTeamRu()) {
            doc.addElement(NOT_SUPPORTED);
            return;
        }

        DocumentSourceInfo documentSourceInfo = getDocumentSourceInfo(request);
        ActualUri uri = webSecurityManager.validateRightAndGetActualUri(documentSourceInfo);

        Option<DataSize> size = storedUriDao.find(uri).getOrThrow(
                () -> new NotFoundException("Couldn't find uri " + uri)
        ).getContentSize();
        DiskResourceId resourceId = diskManager.getDiskResourceId(documentSourceInfo, uri).getOrThrow(
                () -> new BadRequestException("Couldn't find resource id for uri: " + uri));
        Option<MpfsUtils.MpfsOfficeActionCheckResponse> officeActionCheckResponse =
                diskManager.getOfficeRedactorUrl(request.uid, resourceId,
                        Option.ofNullable(request.name), size, Option.ofNullable(request.yandexuid));
        if (officeActionCheckResponse.isPresent()) {
            Element root = doc.addElement("office-online-url");
            root.addText(officeActionCheckResponse.get().officeOnlineUrl);
            Option<String> defaultDownloadsFolder = diskManager.getDefaultDownloadsFolder(request.uid);
            if (defaultDownloadsFolder.isPresent()) {
                root.addAttribute("downloads", defaultDownloadsFolder.get());
            }
            if (officeActionCheckResponse.get().officeOnlineEditorType.isPresent()) {
                root.addAttribute("editor_type", officeActionCheckResponse.get().officeOnlineEditorType.get());
            }
        } else {
            doc.addElement(NOT_SUPPORTED);
        }
    }

    private DocumentSourceInfo getDocumentSourceInfo(OfficeActionCheckRequest request) {
        return DocumentSourceInfo.builder().originalUrl(request.url).uid(request.uid).build().withShowNda(request.showNda);
    }
}
