package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.regex.Matcher;
import java.util.zip.ZipEntry;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultsCleanup;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlSplitter;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.ImageHelper;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfHelper;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ServletResponseOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ForbiddenException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@SuppressWarnings("serial")
@RequiredArgsConstructor
public class HtmlImageAction extends AbstractActionServlet<HtmlImageRequest> implements BackendServlet {

    private static final Logger logger = LoggerFactory.getLogger(HtmlImageAction.class);

    public static final String PATH = "./htmlimage";

    private final DynamicProperty<Boolean> useRemoteImageRendering =
            new DynamicProperty<>("docviewer.image.remote-render", false);

    private final DynamicProperty<Boolean> restoreImagesOnTheFly =
            new DynamicProperty<>("docviewer.image.restore-on-the-fly", false);

    private final boolean enableRemote;

    @Autowired
    private PdfHelper pdfHelper;

    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;

    @Autowired
    private StoredResultHelper storedResultHelper;

    @Autowired
    private TemporaryFileCache temporaryFileCache;

    @Autowired
    private ImageHelper imageHelper;

    @Autowired
    private WebSecurityManager webSecurityManager;

    @Autowired
    private HtmlSplitter htmlSplitter;

    @Autowired
    private MimeDetector mimeDetector;

    @Autowired
    private ResultsCleanup resultsCleanup;

    @Override
    public String getActionUrl() {
        return "/htmlimage";
    }

    @Override
    protected void doGetImpl(final HttpServletRequest req, final HtmlImageRequest request,
            final HttpServletResponse resp)
    {
        if (request.placeholder) {
            getImagePlaceHolder(resp);
        } else {
            getImageFromResult(request, resp);
        }
    }

    private void getImagePlaceHolder(HttpServletResponse resp) {
        resp.setContentType(MimeTypes.MIME_IMAGE_PNG);
        OutputStreamSource outputStreamSource = new ServletResponseOutputStreamSource(resp);
        outputStreamSource.writeFrom(new UrlInputStreamSource(HtmlPostprocessor.IMAGE_PLACEHOLDER_URL));
    }

    private void getImageFromResult(final HtmlImageRequest request, final HttpServletResponse resp)
            throws BadRequestException, ForbiddenException, NotFoundException
    {
        if (StringUtils.isEmpty(request.id)) {
            throw new BadRequestException("File ID is not specified");
        }
        if (StringUtils.isEmpty(request.name)) {
            throw new BadRequestException("File name is not specified");
        }

        webSecurityManager.validateFileRightUsingUid(request.uid, request.id);

        // Make it dynamically. Temporary change. Enable after preview cache full update to PNG.
        // resp.setContentType(detectContentType(request));

        TargetType targetType = request.mobile ? TargetType.HTML_WITH_IMAGES_FOR_MOBILE : TargetType.HTML_WITH_IMAGES;
        StoredResult storedResult = storedResultHelper.getValidatedResult(request.id, targetType);

        if (storedResult.isConvertResultTypePdf()) {
            Matcher matcher = htmlSplitter.getFilenameBackgroundPattern().matcher(request.name);
            if (matcher.matches()) {
                int oneBasedPageIndex = Integer.parseInt(matcher.group(1)) + 1;
                processPdfBackgroundImage(request, oneBasedPageIndex, resp);
            } else {
                throw new BadRequestException("File name does not match background filename pattern");
            }

        } else if (storedResult.isConvertResultTypeZippedHtml()) {
            getImageFromZippedHtml(request, resp, storedResult);
        } else {
            throw new RuntimeException("Unsupported convert result type: " + storedResult.getConvertResultType());
        }

        storedResultHelper.updateResultLastAccess(storedResult);
    }

    private void getImageFromZippedHtml(final HtmlImageRequest request,
            final HttpServletResponse resp, final StoredResult storedResult)
    {
        String fullImageId = storedResult.getFileId() + " " + request.name;

        //retrieve image from cache
        Option<InputStreamSource> temporaryFile = imageHelper.getHtmlBackgroundImageImpl(
                storedResult.getFileId(), request.name);
        if (temporaryFile.isPresent()) {
            logger.debug("Get html image from cache: {}", fullImageId);
            new ServletResponseOutputStreamSource(resp).writeFrom(temporaryFile.get());
            return;
        }

        //first fallback: look for image in archive
        logger.debug("Get html image from zip: {}", fullImageId);
        final FileLink resultFileLink = resultHolder.toFileLink(storedResult.getFileLink().get());
        final File2 resultFile = temporaryFileCache.getOrCreateTemporaryFile(resultFileLink,
                resultHolder::getAsTempFile);
        boolean found = FileUtils.withZipFile(resultFile, zipFile -> {
            ZipEntry entry = zipFile.getEntry(request.name);
            if (entry != null) {
                resp.setContentType(mimeDetector.getMimeType(new ZipEntryInputStreamSource(zipFile, entry)));
                new ServletResponseOutputStreamSource(resp).writeFrom(new ZipEntryInputStreamSource(zipFile, entry));
                return true;
            }
            return false;
        });
        temporaryFileCache.removeFromCache(resultFileLink);

        if (found) {
            return;
        }

        //second fallback: restore image
        if (restoreImagesOnTheFly.get()) {
            logger.info("Try to restore {} image", fullImageId);
            try {
                temporaryFile = imageHelper.restoreImage(storedResult, request.name);
            } catch (Exception e) {
                ExceptionUtils.throwIfUnrecoverable(e);
                logger.error("Got exception while restoring image", e);
            }
            if (temporaryFile.isPresent()) {
                logger.debug("Image restored into cache: {}", fullImageId);
                new ServletResponseOutputStreamSource(resp).writeFrom(temporaryFile.get());
                return;
            }
        }
        logger.info("Failed to resolve image {}, removing result", fullImageId);
        resultsCleanup.cleanupInvalidStoredResult(storedResult);
        throw new NotFoundException("Image was not found for " + fullImageId);
    }

    private void processPdfBackgroundImage(HtmlImageRequest request, int index, HttpServletResponse resp) {
        DimensionO cons = DimensionO.cons(request.width, request.height);
        File2 temporaryFile = (useRemoteImageRendering.get() && enableRemote)?
                              pdfHelper.getHtmlBackgroundImageRemote(request.id, index, cons, request.mobile) :
                              pdfHelper.getHtmlBackgroundImageInplace(request.id, index, cons, request.mobile);

        if (0 < temporaryFile.length() && temporaryFile.length() < Integer.MAX_VALUE) {
            resp.setContentLength((int) temporaryFile.length());
            resp.setContentType(mimeDetector.getMimeType(temporaryFile));
        }

        new ServletResponseOutputStreamSource(resp).writeFrom(temporaryFile);
        temporaryFile.deleteRecursiveQuietly();
    }
}
