package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.zip.ZipEntry;

import org.dom4j.Document;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.pdf.PdfToHtml;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.PageHtmlToResultSerializer;
import ru.yandex.chemodan.app.docviewer.utils.XmlUtils2;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

@SuppressWarnings("serial")
public abstract class HtmlInfoAction extends AbstractXmlActionServlet<ResultRequest> implements BackendServlet {

    @Autowired
    private HtmlPostprocessor htmlPostprocessor;

    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;

    @Autowired
    private StoredResultHelper storedResultHelper;

    private final TargetType targetType;

    @Autowired
    private TemporaryFileCache temporaryFileCache;

    @Autowired
    private WebSecurityManager webSecurityManager;

    @Autowired
    private PdfToHtml pdfToHtml;

    public HtmlInfoAction(TargetType targetType) {
        this.targetType = targetType;
    }

    @Override
    public void execute(final ResultRequest request, final Document doc) {
        if (request.uid == null)
            throw new BadRequestException("User ID ('uid') not specified");

        if (StringUtils.isEmpty(request.id))
            throw new BadRequestException("No file id specified in request");

        webSecurityManager.validateFileRightUsingUid(request.uid, request.id);

        final StoredResult storedResult = storedResultHelper.getValidatedResult(request.id, getConvertTargetMobileIncluded(request));

        final Element docElement = doc.addElement("document");

        XmlUtils2.appendProperties(docElement, storedResult.getDocumentProperties());


        docElement.addAttribute("width-sensitive", String.valueOf(storedResult.isWidthSensitive()));
        final DimensionO size = DimensionO.cons(request.width);

        if (storedResult.isConvertResultTypePdf()) {
            Validate.isTrue(storedResult.getPagesInfo().isPresent(), "Couldn't find info about pages in pdf result");
            Document pageHtml = pdfToHtml.getPageHtml(
                    storedResult.getPagesInfo().get(), Option.empty(), true, size);
            PageHtmlToResultSerializer.addPageHtmlToResult(htmlPostprocessor, xmlSerializer,
                    request, request.id, docElement, size, pageHtml);
        } else if (storedResult.isConvertResultTypeZippedHtml()) {
            final FileLink resultFileLink = resultHolder.toFileLink(storedResult.getFileLink().get());
            final File2 resultSource = temporaryFileCache.getOrCreateTemporaryFile(resultFileLink,
                    resultHolder::getAsTempFile);

            FileUtils.withZipFile(resultSource, zipFile -> {
                    ZipEntry zipEntry = zipFile.getEntry(ConvertToHtmlHelper.ZIPENTRY_NAME_RESULT_XML);

                    if (zipEntry == null) {
                        throw new NotFoundException("Result doesn't have full HTML part");
                    }

                    Document pageHtml = Dom4jUtils
                            .read(new ZipEntryInputStreamSource(zipFile, zipEntry));
                    PageHtmlToResultSerializer.addPageHtmlToResult(htmlPostprocessor, xmlSerializer,
                            request, request.id, docElement, size, pageHtml);
            });

        } else {
            throw new RuntimeException("Unsupported convert result type: " + storedResult.getConvertResultType());
        }

        storedResultHelper.updateResultLastAccess(storedResult);
    }

    @Override
    protected TargetType getConvertTargetMobileIncluded(DocviewerRequest request) {
        TargetType currentTargetType = targetType;
        return currentTargetType == TargetType.HTML_WITH_IMAGES && request.mobile
                ? TargetType.HTML_WITH_IMAGES_FOR_MOBILE
                : currentTargetType;
    }
}
