package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.zip.ZipEntry;

import org.dom4j.Document;
import org.dom4j.Element;
import org.eclipse.jetty.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.pdf.PdfToHtml;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.PageHtmlToResultSerializer;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

@SuppressWarnings("serial")
public abstract class HtmlPageInfoAction extends AbstractPageInfoAction {
    private static final Logger logger = LoggerFactory.getLogger(HtmlPageInfoAction.class);

    @Autowired
    private HtmlPostprocessor htmlPostprocessor;
    @Autowired
    private PdfToHtml pdfToHtml;
    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;
    @Autowired
    private TemporaryFileCache temporaryFileCache;
    @Autowired
    private StoredResultDao storedResultDao;

    private final TargetType targetType;

    public HtmlPageInfoAction(TargetType targetType) {
        this.targetType = targetType;
    }

    @Override
    protected void execute(final PageRequest request, final StoredResult storedResult, final Element pageElement) {
        pageElement.addAttribute("width-sensitive", String.valueOf(storedResult.isWidthSensitive()));
        pageElement.addAttribute("extract-text-info", String.valueOf(storedResult.isConvertResultTypePdf()));

        final DimensionO size = DimensionO.cons(request.width, request.height);

        if (storedResult.isConvertResultTypePdf()) {
            Validate.isTrue(storedResult.getPagesInfo().isPresent(), "Couldn't find info about pages in pdf result");
            Document pageHtml = pdfToHtml.getPageHtml(
                    storedResult.getPagesInfo().get(), Option.of(request.page), true, size);
            PageHtmlToResultSerializer.addPageHtmlToResult(htmlPostprocessor, xmlSerializer,
                    request, request.id, pageElement, size, pageHtml);
        } else if (storedResult.isConvertResultTypeZippedHtml()) {
            final File2 resultFile = getResultFile(storedResult);
            FileUtils.withZipFile(resultFile, zipFile -> {
                    ZipEntry zipEntry = zipFile.getEntry("page_" + request.page + ".xml");

                    if (zipEntry == null)
                        throw new NotFoundException("HTML page " + request.page + " were not generated");

                    Document pageHtml = Dom4jUtils.read(new ZipEntryInputStreamSource(zipFile, zipEntry));
                    PageHtmlToResultSerializer.addPageHtmlToResult(htmlPostprocessor, xmlSerializer,
                            request, request.id, pageElement, size, pageHtml);
            });

        } else {
            throw new RuntimeException("Unsupported convert result type: " + storedResult.getConvertResultType());
        }
    }

    private File2 getResultFile(StoredResult storedResult) {
        final FileLink resultFileLink = resultHolder.toFileLink(storedResult.getFileLink().get());
        try {
            return temporaryFileCache.getOrCreateTemporaryFile(resultFileLink, resultHolder::getAsTempFile);
        } catch (HttpException e) {
            // Temporary hack to fix inconsistent data
            // https://st.yandex-team.ru/DOCVIEWER-1969
            if (e.getStatusCode().isSome(HttpStatus.NOT_FOUND_404)) {
                logger.debug("Remove result meta, because couldn't find result in storage: " + resultFileLink);
                storedResultDao.delete(storedResult.getFileId(), storedResult.getConvertTargetType());
            }
            throw e;
        }
    }

    @Override
    protected TargetType getConvertTargetType() {
        return targetType;
    }
}
