package ru.yandex.chemodan.app.docviewer.web.backend;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.dom4j.Document;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.pdf.PdfToHtml;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.PagesInfoHelper;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.HttpUtils2;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlSerializer;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.FutureX;
import ru.yandex.chemodan.app.docviewer.web.framework.ServletRequestInputStreamSource;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.IoFunction1V;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;

@SuppressWarnings("serial")
public abstract class Post2Action extends HttpServlet implements BackendServlet {

    private static final Logger logger = LoggerFactory.getLogger(Post2Action.class);

    @Autowired
    private ConvertManager convertManager;

    private FileItemFactory fileItemFactory;

    @Autowired
    private HtmlSerializer htmlSerializer;

    @Autowired
    private MimeDetector mimeDetector;

    @Autowired
    private PdfToHtml pdfToHtml;

    @Value("${convert.timeout}")
    private Duration convertTimeout;

    @Value("${docviewer.get.future.result.timeout}")
    private Duration getFutureResultTimeout;

    private final TargetType convertTargetType;

    private final String resultMimeType;

    public Post2Action(TargetType targetType, String resultMimeType) {
        this.convertTargetType = targetType;
        this.resultMimeType = resultMimeType;
    }

    void convert(final File2 temporaryIncomingFile, final String contentType,
            final HttpServletResponse resp)
    {
        logger.info(
                "Processing incoming file of length {} (temporary stored as '{}'). "
                        + "Reported contant type is '{}'",
                temporaryIncomingFile.length(), temporaryIncomingFile.getAbsolutePath(), contentType);

        FileUtils.withEmptyTemporaryFile("result", ".out", temporaryOutputFile -> {
            final String detectedContentType = mimeDetector.getMimeType(temporaryIncomingFile,
                    Collections.singleton(contentType));
            final FutureX<ConvertResultInfo> future = convertManager.scheduleConvert(
                    temporaryIncomingFile, detectedContentType, convertTargetType,
                    new FileOutputStreamSource(temporaryOutputFile));

            logger.debug("Convert were scheduled. Waiting it to be completed...");
            ConvertResultInfo convertResultInfo = future.getUnchecked(convertTimeout.plus(getFutureResultTimeout));

            if (!temporaryOutputFile.exists())
                throw new IllegalStateException(
                        "Convert completed but output file were not created");

            if (!temporaryOutputFile.isRegular() || !temporaryOutputFile.getFile().canRead())
                throw new IllegalStateException(
                        "Convert completed but output file is not a file or can't be read");

            switch (convertTargetType) {
                case HTML_ONLY:
                    if (convertResultInfo.isPdf()) {
                        writeHtmlFromPdf(temporaryOutputFile, resp);
                    } else if (convertResultInfo.isZippedHtml()) {
                        writeFromArchive(temporaryOutputFile, resp);
                    } else {
                        throw new RuntimeException("Unsupported convert result type: " + convertResultInfo.getType());
                    }
                    break;
                case PDF:
                    writeFile(temporaryOutputFile, resp);
                    break;
                case PLAIN_TEXT:
                    writeFile(temporaryOutputFile, resp);
                    break;
                default:
                    throw new UnsupportedOperationException("NYI: " + convertTargetType);
            }
            return null;
        });
    }

    private void convert(final FileItem fileItem, final HttpServletResponse resp) {
        logger.debug("Processing request with file of size {} and content type '{}'",
                fileItem.getSize(), fileItem.getContentType());

        FileUtils.withEmptyTemporaryFile("post-incoming", ".bin", temporaryIncomingFile -> {
            try {
                fileItem.write(temporaryIncomingFile.getFile());
                final String contentType = fileItem.getContentType();

                convert(temporaryIncomingFile, contentType, resp);
            } catch (Exception exc) {
                throw ExceptionUtils.translate(exc);
            }
        });
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) {
        logger.trace("doPost(...)");

        Option<String> contentType = Option.ofNullable(req.getContentType());
        if (contentType.isPresent() && contentType.get().startsWith("multipart/")) {
            doPostMultipart(req, resp);
        } else {
            doPostEntity(req, resp);
        }
    }

    private void doPostEntity(final HttpServletRequest req, final HttpServletResponse resp) {
        FileUtils.withTemporaryFile("post-incoming", ".bin", new ServletRequestInputStreamSource(
                req), temporary -> {
                    convert(temporary, req.getContentType(), resp);
                });
    }

    void doPostMultipart(HttpServletRequest req, HttpServletResponse resp) {
        try {
            final ServletFileUpload servletFileUpload = new ServletFileUpload(fileItemFactory);
            final List<FileItem> fileItems = servletFileUpload.parseRequest(req);

            if (fileItems.isEmpty())
                throw new BadRequestException("No POST data in request");

            try {
                FileItem fileItem = fileItems.get(0);
                convert(fileItem, resp);
            } finally {
                HttpUtils2.cleanupQuietly(fileItems);
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @Override
    public void init() throws ServletException {
        super.init();

        fileItemFactory = new DiskFileItemFactory();
    }

    private void writeFile(final File2 file, HttpServletResponse resp) {
        try {
            if (file.length() < Integer.MAX_VALUE)
                resp.setContentLength((int) file.length());

            resp.setContentType(resultMimeType);
            file.readTo(resp.getOutputStream());
        } catch (IOException exc) {
            throw IoUtils.translate(exc);
        }
    }

    private void writeFromArchive(final File2 file, final HttpServletResponse resp) {
        FileUtils.withZipFile(file, (IoFunction1V<ZipFile>) zipFile -> {
            final ZipEntry zipEntry = zipFile
                    .getEntry(ConvertToHtmlHelper.ZIPENTRY_NAME_RESULT_HTML);
            if (zipEntry == null) {
                throw new UserException(ErrorCode.UNKNOWN_CONVERT_ERROR);
            }

            if (-1 < zipEntry.getSize() && zipEntry.getSize() < Integer.MAX_VALUE)
                resp.setContentLength((int) zipEntry.getSize());

            resp.setContentType(resultMimeType);
            resp.setCharacterEncoding("utf-8");

            new ZipEntryInputStreamSource(zipFile, zipEntry).readTo(resp.getOutputStream());
        });
    }

    private void writeHtmlFromPdf(File2 resultFile, HttpServletResponse resp) {
        try {
            Document doc = PdfUtils.withExistingDocument(resultFile, true, pdDocument -> {
                return pdfToHtml.getPageHtml(
                        PagesInfoHelper.toPagesInfo(pdDocument), Option.empty(), false, DimensionO.WIDTH_900);
            });

            resp.setContentType("text/html");
            resp.setCharacterEncoding("utf-8");

            htmlSerializer.serializeToHtml(doc, resp.getOutputStream());
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

}
