package ru.yandex.chemodan.app.docviewer.web.backend;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.HttpUtils2;
import ru.yandex.chemodan.app.docviewer.web.framework.ServletRequestInputStreamSource;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ReturnHttpCodeException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.TimeUtils;

@SuppressWarnings("serial")
public class Post2IdAction extends HttpServlet implements BackendServlet {

    private static final Logger logger = LoggerFactory.getLogger(Post2IdAction.class);

    static final String PATH = "/post2id";

    private FileItemFactory fileItemFactory;

    @Autowired
    private StateMachine stateMachine;

    @Override
    public String getActionUrl() {
        return PATH;
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        logger.trace("doPost(...)");
        final Instant totalStart = TimeUtils.now();

        try {
            PassportUidOrZero uid = PassportUidOrZero.fromUid(Long.parseLong(StringUtils
                    .trimToEmpty(req.getParameter("uid"))));

            String strType = StringUtils.trimToEmpty(req.getParameter("type"));
            TargetType targetType = TargetType.valueOf(strType, null);
            if (targetType == null)
                throw new BadRequestException("Convert target type ('type') not specified");

            String fileId;
            if (req.getContentType().startsWith("multipart/")) {
                fileId = doPostMultipart(uid, req, targetType, totalStart);
            } else {
                fileId = doPostEntity(uid, req, targetType, totalStart);
            }

            logger.debug("File was submitted using ID '{}'", fileId);

            resp.setContentType("text/plain");
            resp.setCharacterEncoding("utf-8");
            resp.getWriter().write(fileId);
            resp.flushBuffer();

        } catch (ReturnHttpCodeException exc) {
            HttpUtils2.sendError(exc, resp);
        } catch (Exception exc) {
            HttpUtils2.sendError(HttpStatus.SC_500_INTERNAL_SERVER_ERROR, exc, resp);
        }
    }

    private String doPostEntity(final PassportUidOrZero uid, final HttpServletRequest req,
            final TargetType convertTargetType, final Instant totalStart)
    {
        final File2 temporaryFile = FileUtils.createTempFile("submit-post-incoming", ".bin",
                new ServletRequestInputStreamSource(req));
        String fileId = stateMachine.onSubmit(uid, req.getContentType(), temporaryFile,
                convertTargetType, totalStart);
        return fileId;
    }

    private String doPostMultipart(PassportUidOrZero uid, HttpServletRequest req,
            final TargetType convertTargetType, final Instant totalStart)
    {
        try {
            final ServletFileUpload servletFileUpload = new ServletFileUpload(fileItemFactory);
            final List<FileItem> fileItems = servletFileUpload.parseRequest(req);

            if (fileItems.isEmpty())
                throw new BadRequestException("No POST data in request");

            try {
                if (fileItems.size() > 1)
                    logger.warn("Too many POST data in request: {}",
                            fileItems.size());

                FileItem fileItem = fileItems.get(0);
                String fileId = submit(uid, fileItem, convertTargetType, totalStart);
                return fileId;
            } finally {
                HttpUtils2.cleanupQuietly(fileItems);
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    @Override
    public void init() throws ServletException {
        super.init();

        fileItemFactory = new DiskFileItemFactory();
    }

    private String submit(final PassportUidOrZero uid, final FileItem fileItem,
            final TargetType convertTargetType, final Instant totalStart)
    {
        logger.debug("Processing request with file of size {} and content type '{}'",
                fileItem.getSize(), fileItem.getContentType());

        final File2 temporaryFile = FileUtils.createEmptyTempFile("submit-post-incoming", ".bin");
        try {
            fileItem.write(temporaryFile.getFile());

            final String contentType = fileItem.getContentType();
            final String fileId = stateMachine.onSubmit(uid, contentType, temporaryFile,
                    convertTargetType, totalStart);
            return fileId;
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }
}
