package ru.yandex.chemodan.app.docviewer.web.backend;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.function.BiConsumer;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.dom4j.Document;
import org.dom4j.Element;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.ResizeOption;
import ru.yandex.chemodan.app.docviewer.cleanup.ResultsCleanup;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.states.StartManager;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeMalformedURLException;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeURISyntaxException;
import ru.yandex.chemodan.app.docviewer.utils.cache.TemporaryFileCache;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.ImageHelper;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetType;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderer;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.RenderedImageInfo;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.app.docviewer.web.framework.ServletResponseOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.InternalServerErrorException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.UnsupportedMediaTypeException;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

public class PreviewAction extends AbstractActionWithStateValidation<PreviewRequest> implements BackendServlet {

    private static final Logger logger = LoggerFactory.getLogger(PreviewAction.class);
    static final int MAX_PREVIEW_EDGE = 1280;

    @Value("${preview.document.convert.timeout}")
    private Duration convertTimeout;

    @Autowired
    private Scheduler convertScheduler;
    @Autowired
    private StartManager startManager;
    @Autowired
    private StateMachine stateMachine;
    @Autowired
    private ImageHelper imageHelper;
    @Autowired
    private UriHelper uriHelper;
    @Autowired
    private WebSecurityManager webSecurityManager;
    @Autowired
    private StoredResultDao storedResultDao;
    @Autowired
    private PdfRenderer pdfRenderer;
    @Autowired
    private FileStorage resultHolder;
    @Autowired
    private ResultsCleanup resultsCleanup;
    @Autowired
    private TemporaryFileCache temporaryFileCache;
    @Autowired
    private ArchiveManager archiveManager;
    @Autowired
    private RateLimiter rateLimiter;

    @Override
    public String getActionUrl() {
        return "/preview";
    }

    @Override
    protected void doGetImpl(HttpServletRequest req, PreviewRequest request, HttpServletResponse resp) {
        if (rateLimiter.rejectRequest(resp)) {
            return;
        }

        validateRequest(request);

        DocumentSourceInfo source = DocumentSourceInfo.builder().originalUrl(request.url)
                .uid(request.uid).archivePath(StringUtils.notEmptyO(request.archivePath)).build()
                .withShowNda(request.showNda);

        BiConsumer<InputStreamSource, String> handler = (result, type) -> {
            resp.setContentType(type);
            new ServletResponseOutputStreamSource(resp).writeFrom(result);
        };

        getPreview(source, Option.ofNullable(request.width), request.unsafe, Option.ofNullable(request.contentType),
                handler, true);
    }

    public FileLink getPreview(DocumentSourceInfo source, Option<Integer> width, boolean unsafe,
            Option<String> contentType, BiConsumer<InputStreamSource, String> handler, boolean deleteOriginal)
    {
        ActualUri actualUri = uriHelper.rewrite(source);

        State state = convertDocument(source, unsafe, contentType);
        logger.debug("'{}' processing completed with state '{}'", actualUri, state);
        validateState(state, actualUri, TargetType.PREVIEW, "preview generation");

        String fileId = stateMachine.getFileId(actualUri).getOrThrow("File internal id not found");
        StoredResult storedResult = storedResultDao.find(fileId, TargetType.PREVIEW).getOrThrow(() -> {
            throw new IllegalStateException("Stored result must be available");
        });
        ConvertResultType resultType = storedResult.getConvertResultType().getOrThrow(() -> {
            throw new UnsupportedMediaTypeException("Preview is not supported for empty type");
        });

        final FileLink link = resultHolder.toFileLink(storedResult.getFileLink().get());
        try {
            switch (resultType) {
                case PDF: renderPdfFirstPage(link, width, handler); break;
                case ZIPPED_HTML: extractImageFromZippedHtml(link, storedResult, handler); break;
                default: throw new UnsupportedMediaTypeException("Preview is not supported for type " + resultType);
            }
            return link;
        } finally {
            if (deleteOriginal) { temporaryFileCache.removeFromCache(link); }
        }
    }

    public void getPreviewAsync(DocumentSourceInfo source, Option<String> contentType, BiConsumer<File2, String> handler)  {
        convertScheduler.scheduleGlobalTask(source.toString() + contentType, () -> {
            return getPreview(source, Option.empty(), true, contentType, (result, mimeType) -> {
                if (result instanceof File2) {
                    handler.accept((File2) result, mimeType);
                } else {
                    FileUtils.withEmptyTemporaryFile("preview", ".tmp", temp -> {
                        result.readTo(temp);
                        handler.accept(temp, mimeType);
                    });
                }
            }, false);
        }, -2);
    }

    private void validateRequest(PreviewRequest request) throws BadRequestException {
        if (request.uid == null) {
            throw new BadRequestException("No UID is specified");
        } else if (StringUtils.isEmpty(request.url)) {
            throw new BadRequestException("No URL is specified");
        }
    }

    private State convertDocument(DocumentSourceInfo source, boolean isUnsafe, Option<String> mimeType)
            throws BadRequestException
    {
        try {
            if (!isUnsafe) { webSecurityManager.validateUrl(source.getOriginalUrl(), source.getUid()); }
            return startManager.startAndWaitUntilComplete(
                    source,
                    mimeType,
                    TargetType.PREVIEW, convertTimeout, true);
        } catch (RuntimeURISyntaxException | RuntimeMalformedURLException exc) {
            throw new BadRequestException("Specified URL '" + source.getOriginalUrl() + "' is incorrect");
        }
    }

    private void extractImageFromZippedHtml(
            FileLink zipFileLink, StoredResult storedResult, BiConsumer<InputStreamSource, String> handler)
    {
        // should be a zip archive with at most one image within -- see AbstractImageConverter#convertImage
        File2 archiveFile = temporaryFileCache.getOrCreateTemporaryFile(zipFileLink, resultHolder::getAsTempFile);

        ByteArrayOutputStream baos = new ByteArrayOutputStream(512);
        archiveManager.extractOne(archiveFile, ConvertToHtmlHelper.ZIPENTRY_NAME_RESULT_XML, (part) -> baos);

        Document document = Dom4jUtils.read(new ByteArrayInputStream(baos.toByteArray()));

        String imagePath = HtmlPostprocessor.readLocalImageSrc((Element) document.selectSingleNode("//img")).get();
        String mimeType = MimeTypes.IMAGE_EXT_2_MIME.getOrThrow(File2.getExtensionFromPath(imagePath), imagePath);

        Option<InputStreamSource> image = imageHelper.getHtmlBackgroundImageImpl(storedResult.getFileId(), imagePath);

        if (image.isPresent()) {
            logger.debug("Image downloaded using zipped html parsing");
            handler.accept(image.get(), mimeType);
            return;
        }
        Option<InputStreamSource> restored = imageHelper.restoreImage(storedResult, imagePath);

        if (restored.isPresent()) {
            logger.debug("Image from zipped html was restored");
            handler.accept(restored.get(), mimeType);
            return;
        }
        resultsCleanup.cleanupInvalidStoredResult(storedResult);
        throw new InternalServerErrorException("Failed to extract image from zipped html");
    }

    private void renderPdfFirstPage(FileLink pdfFileLink, Option<Integer> width, BiConsumer<InputStreamSource, String> handler) {
        FileUtils.withEmptyTemporaryFile("preview", ".tmp", temp -> {
            ResizeOption resize = width.isPresent() ?
                    ResizeOption.scale(new DimensionO(width, Option.empty())) :
                    ResizeOption.bound(DimensionO.cons(MAX_PREVIEW_EDGE, MAX_PREVIEW_EDGE));

            RenderedImageInfo imageInfo = pdfRenderer.render(pdfFileLink, 1, resize,
                    PdfRenderTargetType.PNG, new FileOutputStreamSource(temp));

            handler.accept(temp, imageInfo.contentType);
        });
    }

}
