package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.dom4j.Document;
import org.jetbrains.annotations.Nullable;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequestWithUid;
import ru.yandex.chemodan.app.docviewer.web.backend.PreviewUrlAction.PreviewUrlRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.PreviewHistoryManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.bazinga.BazingaBender;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.ParallelStreamSupplier;

/**
 * @author vpronto
 */
@SuppressWarnings("serial")
@AllArgsConstructor
@Deprecated
public class PreviewUrlAction extends AbstractXmlActionServlet<PreviewUrlRequest> implements BackendServlet {

    private final static Logger logger = LoggerFactory.getLogger(PreviewUrlAction.class);

    private final static BenderSerializer<PreviewUrlResponse> serializer =
            BazingaBender.mapper.createSerializer(PreviewUrlResponse.class);

    private final DiskManager diskManager;
    private final UriHelper uriHelper;
    private final PreviewHistoryManager previewHistoryManager;
    private final StateMachine stateMachine;
    private final ParallelStreamSupplier parallelStreamSupplier;
    private final PreviewAction previewAction;

    @Override
    public String getActionUrl() {
        return "/preview-url";
    }

    /**
     * @param request
     * @link https://wiki.yandex-team.ru/disk/mpfs/api/json/
     */
    private List<PreviewUrlResponse> execute(PreviewUrlRequest request) {
        ValidateParam.notNull("uid", request.uid);

        PreviewUrlResponse preview = new PreviewUrlResponse("preview");
        if (request.url == null) {
            return Option.of(preview);
        }
        final String parentRequestId = RequestIdStack.current().getOrElse("");
        return parallelStreamSupplier
                .supply(Arrays.stream(request.url.split(","))
                                .map(UrlUtils::urlDecode)
                                .map(item -> new PreviewUrlKey(item, request))
                                .map(r -> () -> {
                                    RequestIdStack.Handle handle = RequestIdStack.pushReplace(parentRequestId);
                                    try {
                                        return getPreview(r);
                                    } finally {
                                        handle.popSafely();
                                    }
                                }),
                        Option.of(preview), Option.of(preview))
                .collect(Collectors.toList());

    }

    protected PreviewUrlResponse getPreview(PreviewUrlKey key) {
        PreviewUrlResponse response = new PreviewUrlResponse(key.url);
        try {
            DocumentSourceInfo source = DocumentSourceInfo.builder()
                    .originalUrl(key.url)
                    .uid(key.uid).build()
                    .withShowNda(key.showNda);
            ActualUri uri = uriHelper.rewrite(source);

            if (diskManager.isMpfsHost(uri)) {
                DiskResourceId diskResourceId = diskManager.getDiskResourceId(source, uri).getOrThrow(
                        () -> new BadRequestException("Uri doesn't correspond to disk resource"));
                Option<String> previewUrl = diskManager.getPreviewUrl(diskResourceId.getServiceFileId(),
                        key.uid,
                        key.size,
                        diskResourceId.getType());

                response.setPreviewUrl(previewUrl);
                response.setState(previewUrl.isPresent() ? State.AVAILABLE : State.NOT_FOUND);
            } else {
                Option<String> preview = previewHistoryManager.resolvePreview(uri, key.uid, key.size);
                response.setPreviewUrl(preview);
                if (!preview.isPresent()) {
                    State state = stateMachine.getState(uri, TargetType.PREVIEW);
                    response.setState(State.CONVERTING);
                    if (state == State.NOT_FOUND || state == State.NOT_STARTED) {
                        logger.warn("Trying to generate on-flight preview for {}", key.url);
                        previewAction.getPreviewAsync(source, Option.empty(),
                                (f, t) -> previewHistoryManager.addPreviewIfAbsent(key.uid, uri, f, t));
                    } else {
                        if (!state.isSubjectTochange()) {
                            throw new IllegalStateException("Can not find preview, state is=" + state);
                        }
                    }
                }
            }
        } catch (Exception e) {
            logger.warn("Can not get preview for {}", key.url, e);
            response.setState(State.NOT_FOUND); //replace to error
        }
        return response;
    }

    @Override
    protected void doGetImpl(HttpServletRequest req, PreviewUrlRequest request, HttpServletResponse resp) {
        try {
            resp.setContentType("application/json");
            resp.getOutputStream().write(serializer.serializeListJson(Cf.toList(execute(request))));
        } catch (Exception exc) {
            logger.warn("Exception occured during action execution", exc);
            resp.setStatus(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
        }
    }

    @Override
    public void execute(PreviewUrlRequest request, Document doc) {  }

    @Data
    @RequiredArgsConstructor
    @BenderBindAllFields
    public static class PreviewUrlResponse {
        final String url;
        Option<String> previewUrl = Option.empty();
        State state = State.AVAILABLE;
    }

    @Data
    public static class PreviewUrlRequest extends DocviewerRequestWithUid {

        @ActionParameter
        @Nullable
        String url;

        @ActionParameter
        @Nullable
        String size;
    }

    @Data
    public static class PreviewUrlKey {

        String url;
        String size;
        String showNda;
        PassportUidOrZero uid;

        public PreviewUrlKey(String url, PreviewUrlRequest request) {
            this.url = url;
            this.size = request.size;
            this.showNda = request.showNda;
            this.uid = request.uid;
        }
    }
}
