package ru.yandex.chemodan.app.docviewer.web.backend;

import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsUtils;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequestWithUid;
import ru.yandex.chemodan.app.docviewer.web.backend.SaveFileFromArchiveAction.SaveFileFromArchiveActionRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author metal
 */
public class SaveFileFromArchiveAction extends AbstractXmlActionServlet<SaveFileFromArchiveActionRequest> implements
        BackendServlet
{
    private static final Logger logger = LoggerFactory.getLogger(SaveFileFromArchiveAction.class);

    public static class SaveFileFromArchiveActionRequest extends DocviewerRequestWithUid {
        @ActionParameter("has-disk")
        boolean hasDisk = true;

        @ActionParameter
        String locale;

        @ActionParameter
        String name;

        @ActionParameter
        String url;

        @ActionParameter("archive-path")
        String archivePath;

        @ActionParameter
        boolean publish = false;
    }

    @Autowired
    private WebSecurityManager webSecurityManager;

    @Autowired
    private DiskManager diskManager;

    @Override
    public String getActionUrl() {
        return "/save-file-from-archive";
    }

    @Override
    public void execute(SaveFileFromArchiveActionRequest request, Document doc) {
        validateParameters(request);

        ActualUri uri = validateRightAndGetActualUri(request);

        DocumentSourceInfo documentSourceInfo = getDocumentSourceInfo(request);
        diskManager.getDiskResourceId(documentSourceInfo, uri).getOrThrow(
                () -> new BadRequestException("Couldn't find attachment path or private hash for url: " + uri));

        if (!request.hasDisk) {
            diskManager.initUser(request.uid, Option.ofNullable(request.locale), request.publish);
        }

        saveFileFromArchiveToDisk(request.uid, doc, documentSourceInfo, StringUtils.notEmptyO(request.name));
    }

    private DocumentSourceInfo getDocumentSourceInfo(SaveFileFromArchiveActionRequest request) {
        String archivePath = StringUtils.startsWith(request.archivePath, "//")
                ? StringUtils.substringAfter(request.archivePath, "//")
                : request.archivePath;
        return DocumentSourceInfo.builder().originalUrl(request.url).uid(request.uid)
                .archivePath(Option.ofNullable(archivePath)).build();
    }

    private ActualUri validateRightAndGetActualUri(SaveFileFromArchiveActionRequest request) {
        DocumentSourceInfo documentSourceInfo =
                DocumentSourceInfo.builder().originalUrl(request.url).uid(request.uid).build()
                        .withShowNda(request.showNda);
        return webSecurityManager.validateRightAndGetActualUri(documentSourceInfo);
    }

    private void saveFileFromArchiveToDisk(PassportUidOrZero uid, Document doc,
            DocumentSourceInfo documentSourceInfo, Option<String> name)
    {
        try {
            String defaultDownloadsFolder = diskManager.getDefaultDownloadsFolder(uid).get();

            String operationId = diskManager.extractAndSaveFileFromArchiveToDiskAndGetOperationId(uid,
                    documentSourceInfo, name);

            Option<String> oid = Option.of(operationId);
            Element root = doc.addElement("ok");
            if (oid.isPresent()) {
                root.addElement("oid").addText(oid.get());
            }
            Element downloads = root.addElement("downloads");
            downloads.addText(defaultDownloadsFolder);
        } catch (HttpException e) {
            if (e.getStatusCode().isSome(MpfsUtils.HTTP_ERROR_OUT_OF_SPACE)) {
                logger.debug("Couldn't extract and save file, out of space");
                doc.addElement("out-of-space");
            } else {
                throw new RuntimeException("Couldn't extract and save file", e);
            }
        }
    }

    private void validateParameters(SaveFileFromArchiveActionRequest request) throws BadRequestException {
        if (StringUtils.isEmpty(request.url)) {
            throw new BadRequestException("No file url specified in request");
        }
        if (!request.uid.isAuthenticated()) {
            throw new BadRequestException("No auth");
        }
        if (StringUtils.isEmpty(request.archivePath)) {
            throw new BadRequestException("No archive path specified in request");
        }
    }
}
