package ru.yandex.chemodan.app.docviewer.web.backend;

import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.disk.mpfs.MpfsUtils;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceType;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequestWithUid;
import ru.yandex.chemodan.app.docviewer.web.backend.SaveToDiskAction.SaveToDiskRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractXmlActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author akirakozov
 * TODO: support preview copy
 */
@SuppressWarnings("serial")
public class SaveToDiskAction extends AbstractXmlActionServlet<SaveToDiskRequest> implements BackendServlet {
    private static final Logger logger = LoggerFactory.getLogger(SaveToDiskAction.class);

    public static class SaveToDiskRequest extends DocviewerRequestWithUid {
        @ActionParameter("has-disk")
        boolean hasDisk = true;

        @ActionParameter
        String locale;

        @ActionParameter
        String name;

        @ActionParameter(required = true)
        String url;

        @ActionParameter
        boolean publish = false;
    }

    @Autowired
    private WebSecurityManager webSecurityManager;
    @Autowired
    private DiskManager diskManager;

    @Override
    public String getActionUrl() {
        return "/save-to-disk";
    }

    @Override
    public void execute(SaveToDiskRequest request, Document doc) {
        validateParameters(request);

        DocumentSourceInfo documentSourceInfo = getDocumentSourceInfo(request);
        ActualUri uri = validateRightAndGetActualUri(request);

        DiskResourceId resourceId = diskManager.getDiskResourceId(documentSourceInfo, uri).getOrThrow(
                () -> new BadRequestException("Couldn't find attachment path or private hash for url: " + uri));

        if (!request.hasDisk) {
            diskManager.initUser(request.uid, Option.ofNullable(request.locale), request.publish);
        }

        copyToDisk(request.uid, doc, resourceId, StringUtils.notEmptyO(request.name),
                request.publish, NdaUtils.isShowNdaSet(request.showNda));
    }

    private DocumentSourceInfo getDocumentSourceInfo(SaveToDiskRequest request) {
        return DocumentSourceInfo.builder().originalUrl(request.url).uid(request.uid).build().withShowNda(request.showNda);
    }

    private ActualUri validateRightAndGetActualUri(SaveToDiskRequest request) {
        return webSecurityManager.validateRightAndGetActualUri(getDocumentSourceInfo(request));
    }

    private void copyToDisk(PassportUidOrZero uid, Document doc,
            DiskResourceId resourceId, Option<String> name, boolean publish, boolean showNda)
    {
        try {
            Option<String> oid = Option.empty();
            if (publish && resourceId.getType() == DiskResourceType.MAIL_ATTACHMENT) {
                oid = Option.of(diskManager.asyncCopyToDiskAndGetOperationId(uid, resourceId, name, true, showNda));
            } else if (resourceId.getType().isAsyncCopySupported()) {
                oid = Option.of(diskManager.asyncCopyToDiskAndGetOperationId(uid, resourceId, name, false, showNda));
            } else {
                diskManager.copyResourceToDisk(uid, resourceId, name, false, showNda);
            }
            Element root = doc.addElement("ok");
            if (oid.isPresent()) {
                root.addElement("oid").addText(oid.get());
            }

            Element downloads = root.addElement("downloads");
            downloads.addText(diskManager.getDefaultDownloadsFolder(uid).get());
        } catch (HttpException e) {
             if (e.getStatusCode().isSome(MpfsUtils.HTTP_ERROR_OUT_OF_SPACE)) {
                logger.debug("Couldn't copy file, out of space");
                doc.addElement("out-of-space");
            } else {
                throw new RuntimeException("Couldn't copy file", e);
            }
        }
    }

    private void validateParameters(SaveToDiskRequest request) throws BadRequestException {
        if (!request.uid.isAuthenticated()) {
            throw new BadRequestException("No auth");
        }
    }
}
