package ru.yandex.chemodan.app.docviewer.web.backend;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.crypt.TokenManager;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.states.FileTooBigUserException;
import ru.yandex.chemodan.app.docviewer.states.PasswordProtectedException;
import ru.yandex.chemodan.app.docviewer.utils.XmlSerializer;
import ru.yandex.chemodan.app.docviewer.web.backend.SourceAction.SourceRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.ActionParameter;
import ru.yandex.chemodan.app.docviewer.web.framework.XmlActionUtils;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ForbiddenException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@SuppressWarnings("serial")
public class SourceAction extends AbstractSourceAction<SourceRequest> implements BackendServlet {
    private static final Logger logger = LoggerFactory.getLogger(SourceAction.class);
    private static final String X_ACCEL_REDIRECT_HEADER = "X-Accel-Redirect";

    public static class SourceRequest extends AbstractSourceRequest {
        @ActionParameter("archive-path")
        @Nullable
        private String archivePath;

        @ActionParameter
        @Nullable
        private String id;

        @ActionParameter("ts")
        String hexTs;

        @ActionParameter
        String token;

        @ActionParameter
        @Nullable
        private String url;

        @ActionParameter
        private String session;

        @Override
        Option<String> getSession() {
            return StringUtils.notEmptyO(session);
        }

        @Override
        Option<String> getArchivePath() {
            return StringUtils.notEmptyO(archivePath);
        }

        @Override
        Option<String> getUrl() {
            return StringUtils.notEmptyO(url);
        }

        @Override
        Option<String> getFileId() {
            return StringUtils.notEmptyO(id);
        }
    }

    @Autowired
    protected XmlSerializer xmlSerializer;

    @Autowired
    private TokenManager tokenManager;

    @Override
    public String getActionUrl() {
        return "/source";
    }

    @Override
    protected void doGetImpl(HttpServletRequest req, SourceRequest request, HttpServletResponse resp) {
        validateToken(request, request.getArchivePath());

        try {
            extractAndWriteResult(request, req, resp);
        } catch (FileTooBigUserException ftb) {
            logger.error("User requested too big file, actual: {}, limit: {}",
                    ftb.getActualLength(), ftb.getMaxLength(), ftb);
            resp.setStatus(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
            Document result = DocumentHelper.createDocument();
            Element root = result.addElement("file-too-big");
            root.addElement("limit-length").addText(Long.toString(ftb.getMaxLength()));
            XmlActionUtils.serializeAndWriteResult(xmlSerializer, resp, result, request.json);
        } catch (PasswordProtectedException ppe) {
            Check.some(ppe.getErrorArchivePath());
            try {
                resp.setStatus(HttpStatus.SC_204_NO_CONTENT);
                resp.setHeader(X_ACCEL_REDIRECT_HEADER,
                        UrlUtils.addParameter("/password.xml", "archive-path", ppe.getErrorArchivePath().get()));
            } catch (Exception exc2) {
                throw ExceptionUtils.translate(exc2);
            }
        }
    }


    @Override
    protected StoredUri findFileUri(SourceRequest request) {
        StoredUri storedUri = super.findFileUri(request);
        ActualUri actualUri = storedUri.getUri();

        validateDiskUrlForArchive(actualUri, request.getArchivePath());

        return storedUri;
    }

    private void validateToken(SourceRequest request, Option<String> archivePath) {
        if (request.token == null) {
            throw new ForbiddenException("Request without token");
        }

        try {
            tokenManager.validateToken(request.token, request.hexTs, Option.ofNullable(request.id),
                    request.getUrl(), archivePath, request.uid);
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Illegal token or timestamp: " + e.getMessage());
        }
    }

    private void validateDiskUrlForArchive(ActualUri actualUri, Option<String> archivePath) {
        boolean isMpfsHost = mpfsHost.equals(actualUri.getUri().getHost());
        boolean isFileInArchive = archivePath.isPresent();

        if (isMpfsHost && !isFileInArchive) {
            throw new BadRequestException("Source action shouldn't be used for disk urls w/o archive path");
        }
    }
}
