package ru.yandex.chemodan.app.docviewer.web.backend;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.Data;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.disk.resource.DiskResourceId;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeMalformedURLException;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeURISyntaxException;
import ru.yandex.chemodan.app.docviewer.utils.XmlSerializer;
import ru.yandex.chemodan.app.docviewer.utils.scheduler.Scheduler;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.XmlActionUtils;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
@Data
public class StartAction extends AbstractActionServlet<StartRequest> implements BackendServlet {
    private static final Logger logger = LoggerFactory.getLogger(StartAction.class);

    static final String PATH = "/start";

    @Autowired
    private StateMachine stateMachine;
    @Autowired
    private DiskManager diskManager;
    @Autowired
    private WebSecurityManager webSecurityManager;
    @Autowired
    protected XmlSerializer xmlSerializer;
    @Autowired
    private Scheduler copierScheduler;
    @Autowired
    private RateLimiter rateLimiter;

    @Override
    protected void doGetImpl(HttpServletRequest req, StartRequest request, HttpServletResponse resp) {
        if (rateLimiter.rejectRequest(resp)) {
            return;
        }
        Document result;
        Document doc = DocumentHelper.createDocument();
        execute(req, request, doc, resp);
        result = doc;

        XmlActionUtils.serializeAndWriteResult(xmlSerializer, resp, result, request.json);
    }

    protected void execute(HttpServletRequest req, StartRequest request, Document doc, HttpServletResponse resp) {
        logger.trace("execute {}(uid: '{}', url: '{}', type: '{}')",
                getActionUrl(), request.url, request.uid, request.type);

        executeInner(request);

        doc.addElement("ok");
    }

    private void executeInner(StartRequest request) {
        try {
            if (!request.unsafe) {
                webSecurityManager.validateUrl(request.url, request.uid, Option.ofNullable(request.serpParams));
            }

            // only start document handling
            stateMachine.onStart(getDocumentSourceInfo(request), getActionUrl(),
                    Option.empty(), request.type,
                    request.session, parseSkipableContentTypes(request.skipMimes),
                    getStartRequestTime(), false);

        } catch (RuntimeMalformedURLException | RuntimeURISyntaxException exc) {
            throw new BadRequestException("Specified URL '" + request.url + "' is incorrect");
        }
    }

    protected DocumentSourceInfo getDocumentSourceInfo(StartRequest request) {
        DocumentSourceInfo info = DocumentSourceInfo.builder().originalUrl(request.url).uid(request.uid)
                .archivePath(StringUtils.notEmptyO(request.archivePath))
                .showNda(NdaUtils.isShowNdaSet(request.showNda))
                .warmUp(request.warmUp).build();
        return info.toBuilder().remoteFileId(getHidSafe(info)).build();
    }

    private Option<String> getHidSafe(DocumentSourceInfo info) {
        Option<DiskResourceId> diskResourceIdO = diskManager.getDiskResourceId(info);
        if (diskResourceIdO.isPresent() && !info.getArchivePath().isPresent()) {
            DiskResourceId diskResourceId = diskResourceIdO.get();
            switch (diskResourceId.getType()) {
                case DISK_PUBLIC_FILE:
                case DISK_PRIVATE_VERSIONING_FILE:
                case DISK_PRIVATE_FILE:
                    try {
                        Option<String> resourceId = diskManager.getHid(info.getUid(), diskResourceId);
                        resourceId.ifPresent(r -> logger.info("Enriched remote resource info {} {}", r, diskResourceId));
                        return resourceId;
                    } catch (HttpException he) {
                        if (he.statusCodeIs(HttpStatus.SC_403_FORBIDDEN)) {
                            throw new UserException(ErrorCode.FILE_IS_FORBIDDEN);
                        }
                        logger.error("Can't fetch hid for {}", diskResourceId, he);
                    } catch (Exception e) {
                        logger.error("Can't fetch hid for {}", diskResourceId, e);
                    }
                default:
                    return Option.empty();

            }

        }
        return Option.empty();
    }

    public static ListF<String> parseSkipableContentTypes(String skipMimes) {
        return Cf.list(StringUtils.split(Option.ofNullable(skipMimes).getOrElse(""), ","));
    }

    @Override
    public String getActionUrl() {
        return PATH;
    }

    protected Instant getStartRequestTime() {
        return Instant.now();
    }
}
