package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.dao.pdfImage.ImageDao;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.StoredImage;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.NotFoundException;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;

/**
 * @author vlsergey
 * @author akirakozov
 */
public class StoredResultHelper {

    private static final Duration DURATION_TO_UPATE = Duration.standardMinutes(5);
    private final Executor executor = Executors.newCachedThreadPool(new ThreadNameIndexThreadFactory("cache-update"));

    @Autowired
    private StoredResultDao storedResultDao;

    @Autowired
    @Qualifier("pdfImageDao")
    private ImageDao pdfImageDao;

    private static boolean needUpdate(Instant lastAccess) {
        return lastAccess.plus(DURATION_TO_UPATE).isBeforeNow();
    }

    StoredResult getValidatedResult(String fileId, TargetType targetType) {
        StoredResult storedResult = storedResultDao.find(fileId, targetType)
                .getOrThrow(() -> new NotFoundException("File specified were not converted"));

        storedResult.getError().ifPresent(error -> {
            throw new BadRequestException("Conversion result can't be obtained because of error: " + error);
        });

        if (!storedResult.getFileLink().isPresent()) {
            throw new NotFoundException("Conversion result is not ready yet or already deleted");
        }

        return storedResult;
    }

    void validatePage(StoredResult result, int page) {
        int totalPages = result.getPages().getOrElse(Integer.MAX_VALUE);

        if (1 > page || page > totalPages) {
            throw new NotFoundException("Page index '" + page + "' is out of bound '[1; "+ totalPages + "]'");
        }
    }

    void updateResultLastAccess(StoredResult result) {
        CompletableFuture.runAsync(() -> {
            if (needUpdate(result.getLastAccess())) {
                storedResultDao.updateLastAccessTime(result.getFileId(), result.getConvertTargetType());
            }
        }, executor);
    }

    //TODO: improve it
    public void updatePdfImageLastAccess(StoredImage pdfImage) {
        CompletableFuture.runAsync(() -> {
            if (needUpdate(pdfImage.lastAccess)) {
                pdfImageDao.updateLastAccessTime(pdfImage.fileId, pdfImage.pageIndex, pdfImage.size);
            }
        }, executor);
    }

    public void savePdfImageIfNotExists(String fileId, int oneBasedPageIndex, DimensionO size, String serializedPath) {
        CompletableFuture.runAsync(() -> pdfImageDao.savePdfImageIfNotExists(fileId, oneBasedPageIndex, size, serializedPath), executor);
    }
}
