package ru.yandex.chemodan.app.docviewer.web.backend;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.states.StartManager;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.states.StateMachine;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeMalformedURLException;
import ru.yandex.chemodan.app.docviewer.utils.RuntimeURISyntaxException;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.InternalServerErrorException;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

@SuppressWarnings("serial")
public class Url2IdAction extends AbstractActionServlet<Url2IdRequest> implements BackendServlet {

    public static final String PATH = "/url2id";

    @Autowired
    private StartManager startManager;

    @Autowired
    private StateMachine stateMachine;

    @Value("${actions.url2id.timeout}")
    private Duration timeout;

    @Autowired
    private WebSecurityManager webSecurityManager;

    @Autowired
    private UriHelper uriHelper;

    @Override
    public String getActionUrl() {
        return PATH;
    }

    @Override
    protected void doGetImpl(HttpServletRequest req, Url2IdRequest request, HttpServletResponse resp) {
        if (request.uid == null) {
            throw new BadRequestException("No UID is specified");
        } else if (StringUtils.isEmpty(request.url)) {
            throw new BadRequestException("No URL is specified");
        } else if (request.type == null) {
            throw new BadRequestException("Convert target type is not specified");
        } else {
            String fileId = url2id(request);
            HttpServletResponseX.wrap(resp).writeContent(fileId.getBytes(), "text/plain; charset=utf-8");
        }
    }

    private String url2id(Url2IdRequest request) {
        try {
            if (!request.unsafe) {
                webSecurityManager.validateUrl(request.url, request.uid);
            }

            DocumentSourceInfo source = DocumentSourceInfo.builder().originalUrl(request.url)
                    .uid(request.uid).archivePath(StringUtils.notEmptyO(request.archivePath)).build()
                    .withShowNda(request.showNda);

            ActualUri actualUri = uriHelper.rewrite(source);

            // TODO: add request.contentType
            State state = startManager.startAndWaitUntilComplete(
                    source, Option.ofNullable(request.contentType),
                    request.type, timeout, request.complete);

            Option<String> fileId = stateMachine.getFileId(actualUri);
            if (fileId.isPresent()) {
                if (request.complete) {
                    Validate.isTrue(state == State.AVAILABLE, "Document result isn't available");
                }
                return fileId.get();
            } else {
                throw new InternalServerErrorException(
                        "State '" + state + "' is final and no file ID assigned to URL");
            }
        } catch (RuntimeURISyntaxException | RuntimeMalformedURLException exc) {
            throw new BadRequestException("Specified URL '" + request.url + "' is incorrect");
        }
    }

}
