package ru.yandex.chemodan.app.docviewer.web.backend;

import java.util.Collections;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.client.methods.HttpGet;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.html.ConvertToHtmlHelper;
import ru.yandex.chemodan.app.docviewer.web.framework.AbstractActionServlet;
import ru.yandex.chemodan.app.docviewer.web.framework.ServletResponseOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.web.framework.WebSecurityManager;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.misc.io.IoFunction1V;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClient4Utils;
import ru.yandex.misc.lang.StringUtils;

@SuppressWarnings("serial")
public class Url2ResultAction extends AbstractActionServlet<StartRequest> implements BackendServlet {

    @Autowired
    private ConvertManager convertManager;

    @Autowired
    private MimeDetector mimeDetector;

    @Value("${actions.url2result.timeout}")
    private Duration timeout;

    @Autowired
    private WebSecurityManager webSecurityManager;

    @Override
    public String getActionUrl() {
        return "/url2result";
    }

    @Override
    protected void doGetImpl(final HttpServletRequest req, final StartRequest request,
            final HttpServletResponse resp)
    {
        webSecurityManager.validateUrl(request.url, request.uid);

        final TargetType targetType = request.type;
        if (targetType == null)
            throw new BadRequestException("Convert target type is not specified");

        if (StringUtils.isEmpty(request.url))
            throw new BadRequestException("No URL is specified");

        FileUtils.withEmptyTemporaryFile("urlaction", ".in", new IoFunction1V<File2>() {
            @Override
            public void applyWithException(final File2 temporaryInputFile) throws Exception {

                HttpGet httpGet = new HttpGet(request.url);
                final String reportedContentType = ApacheHttpClient4Utils.execute(httpGet,
                        response -> {
                            final String contentType = response.getEntity().getContentType()
                                    .getValue();
                            new FileOutputStreamSource(temporaryInputFile).writeFrom(response
                                    .getEntity().getContent());
                            return contentType;
                        }, Timeout.valueOf(getTimeout()));

                final String detectedContentType = mimeDetector.getMimeType(temporaryInputFile,
                        Collections.singleton(reportedContentType));

                switch (targetType) {
                    case HTML_ONLY: {
                        FileUtils.withEmptyTemporaryFile("urlaction", ".zip",
                                (IoFunction1V<File2>) temporaryOutputFile -> {
                                    Future<?> future = convertManager.scheduleConvert(
                                            temporaryInputFile, detectedContentType,
                                            targetType, new FileOutputStreamSource(
                                                    temporaryOutputFile));
                                    future.get();

                                    FileUtils.withZipFile(temporaryOutputFile,
                                            (IoFunction1V<ZipFile>) zipFile -> {
                                                ZipEntry zipEntry = zipFile
                                                        .getEntry(ConvertToHtmlHelper.ZIPENTRY_NAME_RESULT_HTML);
                                                resp.setContentLength((int) zipEntry
                                                        .getSize());
                                                resp.setContentType(MimeTypes.MIME_HTML
                                                        + "; charset=utf-8");
                                                new ServletResponseOutputStreamSource(resp)
                                                        .writeFrom(new ZipEntryInputStreamSource(
                                                                zipFile, zipEntry));
                                            });
                                });
                        break;
                    }
                    case PDF:
                        resp.setContentType(MimeTypes.MIME_PDF);
                        break;
                    case HTML_WITH_IMAGES:
                    case HTML_WITH_IMAGES_FOR_MOBILE:
                        resp.setContentType(MimeTypes.MIME_ARCHIVE_ZIP);
                        break;
                    case PLAIN_TEXT:
                        resp.setContentType("text/plain; charset=utf-8");
                        break;
                    default:
                        // no op;
                        break;
                }

                if (targetType != TargetType.HTML_ONLY) {
                    convertAndOutputResult(temporaryInputFile, detectedContentType, resp);
                }
            }

            private void convertAndOutputResult(final File2 temporaryInputFile,
                    String detectedContentType, final HttpServletResponse response)
                    throws InterruptedException, ExecutionException
            {
                Future<?> future = convertManager.scheduleConvert(temporaryInputFile,
                        detectedContentType, targetType, new ServletResponseOutputStreamSource(
                                response));
                future.get();
            }
        });
    }

    public Duration getTimeout() {
        return timeout;
    }

    public void setTimeout(Duration timeout) {
        this.timeout = timeout;
    }
}
