package ru.yandex.chemodan.app.docviewer.web.client;

import java.io.IOException;

import javax.annotation.PreDestroy;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.util.EntityUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.CopyInfo;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.httpclient.DotReplacingHostnameProcessor;
import ru.yandex.chemodan.app.docviewer.web.backend.PrerenderHtmlImageAction;
import ru.yandex.chemodan.app.docviewer.web.backend.RenderImageAction;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.InstrumentedCloseableHttpClient;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 * @author swined
 */
public class DocviewerForwardClient {

    private static final Logger logger = LoggerFactory.getLogger(DocviewerForwardClient.class);

    public static final String AUTH_HEADER = "Authorization";

    private final DynamicProperty<Boolean> uniformForward = new DynamicProperty<>("uniform-forward", false);

    private final String docviewerUrl;
    private final String authKey;
    private final HttpClient httpClient;

    public DocviewerForwardClient(
            String docviewerUrl,
            String authKey,
            HttpClient httpClient)
    {
        this.docviewerUrl = docviewerUrl;
        this.authKey = authKey;
        this.httpClient = httpClient;

        ((InstrumentedCloseableHttpClient) httpClient)
                .setHostnameProcessor(new DotReplacingHostnameProcessor());
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    public void forwardToStartInternal(CopyInfo info, Option<String> resultContentType) {

        MapF<String, Object> parameters = Cf.<String, Object>map("url", info.getStartInfo().getSource().getOriginalUrl())
                .plus1("uid", info.getSource().getUid())
                .plus1("type", info.getStartInfo().getConvertTargetType())
                .plus1("start-time", info.getStartInfo().getStartTime())
                .plus1("warm-up", info.getStartInfo().getSource().isWarmUp());
        if (info.getSource().isShowNda()) {
            parameters = parameters.plus1("show_nda", "1");
        }
        parameters = addOptionalParameter(parameters, "content-type", resultContentType);
        parameters = addOptionalParameter(parameters, "archive-path", info.getSource().getArchivePath());
        if (!info.getStartInfo().getSkipableContentTypes().isEmpty()) {
            parameters = parameters.plus1("skip-mimes", String.join(", ", info.getStartInfo().getSkipableContentTypes()));
        }

        String uriString = UrlUtils.addParameters(docviewerUrl + "/start-internal", parameters);
        HttpGet request = new HttpGet(uriString);

        if (StringUtils.isNotBlank(authKey)) {
            request.setHeader(AUTH_HEADER, "Basic " + authKey);
        }

        String originalUrl = info.getSource().getOriginalUrl();

        String hashSource = uniformForward.get()
                ? Md5.A.digest(originalUrl).hex()
                : originalUrl;

        request.setHeader("X-Req-Id", String.valueOf(hashSource.hashCode()));
        exec(request);
    }

    public void forwardToPrerenderHtmlImage(String fileId, DimensionO size, int oneBasedPageIndex, boolean mobile) {
        MapF<String, Object> parameters = Cf.<String, Object>map()
                .plus1("fileId", fileId)
                .plus1("width", size.width.get())
                .plus1("height", size.height.get())
                .plus1("oneBasedPageIndex", oneBasedPageIndex)
                .plus1("mobile", mobile);
        exec(new HttpGet(UrlUtils.addParameters(docviewerUrl + PrerenderHtmlImageAction.URL, parameters)));
    }

    public String forwardToPrerenderSingleHtmlImage(String fileId, DimensionO size, int oneBasedPageIndex, boolean mobile) {
        MapF<String, Object> parameters = Cf.<String, Object>map()
                .plus1("id", fileId)
                .plus1("width", size.width.get())
                .plus1("index", oneBasedPageIndex)
                .plus1("mobile", mobile)
                .plus1("json", 1);
        parameters = size.height.isPresent() ? parameters.plus1("height", size.height.get()) : parameters;
        String link =
                exec(new HttpGet(UrlUtils.addParameters(docviewerUrl + RenderImageAction.URL, parameters)));
        return JsonNodeUtils.getNode(link).get("link").textValue();
    }

    private MapF<String, Object> addOptionalParameter(MapF<String, Object> currentParams,
            String key, Option<String> value)
    {
        return value.isPresent() && StringUtils.isNotBlank(value.get())
                ? currentParams.plus1(key, value.get())
                : currentParams;
    }

    private String exec(HttpUriRequest request) {
        return RetryUtils.retryOrThrow(logger, 3, () -> {
            try {
                HttpResponse response = httpClient.execute(request);
                int code = response.getStatusLine().getStatusCode();
                if (code != HttpStatus.SC_OK) {
                    String error = String.format("Could not execute method '%s'. Server returned %s",
                            request, response.getStatusLine().getReasonPhrase());
                    throw new HttpException(code, EntityUtils.toString(response.getEntity()), error);
                }
                return EntityUtils.toString(response.getEntity());
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }

}
