package ru.yandex.chemodan.app.docviewer.web.framework;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.util.Map;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.HttpUtils2;
import ru.yandex.chemodan.app.docviewer.web.DocviewerRequest;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.BadRequestException;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ReturnHttpCodeException;
import ru.yandex.chemodan.app.docviewer.web.framework.tvm.Tvm2Utils;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.inside.passport.tvm2.UserTicketHolder;
import ru.yandex.inside.passport.tvm2.exceptions.IncorrectTvmUserTicketException;
import ru.yandex.inside.passport.tvm2.exceptions.TvmBaseException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;
import ru.yandex.misc.reflection.GenericUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

@SuppressWarnings("serial")
public abstract class AbstractActionServlet<T extends DocviewerRequest> extends HttpServlet {

    private static final Logger logger = LoggerFactory.getLogger(AbstractActionServlet.class);

    private final static MapF<ErrorCode, Integer> httpCodes = Cf.map(
            ErrorCode.FILE_IS_FORBIDDEN, HttpServletResponse.SC_FORBIDDEN,
            ErrorCode.FILE_NOT_FOUND, HttpServletResponse.SC_NOT_FOUND);

    private Constructor<T> toFillConstructor;
    private Map<String, FieldInfo> toFillFields;

    @Autowired
    private Tvm2 tvm;

    @Value("${tvm.external.enabled}")
    private boolean externalTvmAuthenticationEnabled;

    public AbstractActionServlet() {
        initRequestConstructor();
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) {
        try {
            T request = null;
            try {
                checkTvmAuthentication(req);
                request = parseRequest(req, resp);
                logger.debug("Processing {} with {}",
                        getClass().getSimpleName(),
                        YandexToStringBuilder.reflectionToStringValueObject(request));

                Option<String> tvmUserTicketO = Option.ofNullable(req.getHeader(TvmHeaders.USER_TICKET));
                T finalRequest = request;
                UserTicketHolder.withUserTicketO(tvmUserTicketO, () -> doGetImpl(req, finalRequest, resp));

            } catch (HttpException he) {
                Integer code = he.getStatusCode().getOrElse(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
                logSendingError(code, req, he);
                sendError(code, he, resp, Option.ofNullable(request));
            } catch (IllegalArgumentException ie) {
                logSendingError(HttpStatus.SC_400_BAD_REQUEST, req, ie);
                sendError(HttpStatus.SC_400_BAD_REQUEST, ie, resp, Option.ofNullable(request));
            } catch (FoundException exc) {
                logger.debug("Redirecting to '{}' in response to '{}'...",
                        exc.getUri(),
                        HttpServletRequestX.wrap(req).getTrueRequestUrl());
                HttpUtils2.sendRedirect(exc, resp);

            } catch (ReturnHttpCodeException exc) {
                logSendingError(exc.getStatusCode(), req, exc);
                sendError(exc.getStatusCode(), exc, resp, Option.ofNullable(request));

            } catch (UserException exc) {
                final ErrorCode errorCode = exc.getErrorCode();
                if (httpCodes.containsKeyTs(errorCode)) {
                    final int httpErrorCode = httpCodes.getTs(errorCode);
                    logSendingError(httpErrorCode, req, exc);
                    Exception excWithHiddenMessage = new Exception("Error: " + httpErrorCode);
                    sendError(httpErrorCode, excWithHiddenMessage, resp, Option.ofNullable(request));
                } else {
                    throw exc;
                }
            }

        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    protected void sendError(final int code, Throwable he, HttpServletResponse resp, Option<T> req) throws IOException {
        HttpUtils2.sendError(code, he, resp);
    }

    private void checkTvmAuthentication(HttpServletRequest req) {
        try {
            if (isTvmAuthEnabled()) {
                Tvm2Utils.checkAuthentication(tvm, req);
            }
        } catch (TvmBaseException e) {
            if (EnvironmentType.getActive() == EnvironmentType.TESTING
                    && e instanceof IncorrectTvmUserTicketException
                    && Option.ofNullable(req.getParameter("uid"))
                    .filterNot(String::isEmpty).map(Long::valueOf)
                    .filter(uid -> PassportUidOrZero.fromUid(uid).isYandexTeamRu()).isPresent())
            {
                // DOCVIEWER-2518
                logger.warn(e.getMessage() + " Ignored for yandex-team users in testing");
                return;
            }
            logger.warn(e.getMessage());
            throw new UserException(ErrorCode.FILE_IS_FORBIDDEN, e.getMessage());
        }
    }

    protected boolean isTvmAuthEnabled() {
        return externalTvmAuthenticationEnabled;
    }

    private void logSendingError(int httpErrorCode, HttpServletRequest req, Throwable exc) {
        String message = String.format("Sending error with code %d in response to '%s'...",
                httpErrorCode, HttpServletRequestX.wrap(req).getTrueRequestUrl());
        logger.error(message, exc);
    }

    protected abstract void doGetImpl(HttpServletRequest req, T request,
            HttpServletResponse resp);

    protected Class<T> getRequestClass() {
        return (Class<T>) GenericUtils.getSingleSpecializationParameter(getClass());
    }

    protected void initRequestConstructor() {
        Class<T> requestClass = getRequestClass();

        if (requestClass == null) {
            throw new RuntimeException("Class '" + this.getClass().getName()
                    + "' doesn't declare method 'execute' with required signature");
        }

        try {
            this.toFillConstructor = requestClass.getConstructor();
        } catch (NoSuchMethodException exc) {
            throw new RuntimeException("Class '" + requestClass
                    + "' doesn't declare public or protected constructor without arguments");
        }

        Map<String, FieldInfo> toFillNew = Cf.hashMap();

        Class<?> toAnalyze = requestClass;
        while (toAnalyze != null) {
            for (Field field : toAnalyze.getDeclaredFields()) {

                ActionParameter annotation = field.getAnnotation(ActionParameter.class);
                if (annotation == null) {
                    continue;
                }

                FieldInfo fieldInfo = new FieldInfo(field, annotation.required());
                field.setAccessible(true);

                if (ServletRequest.class.isAssignableFrom(field.getType())) {

                    toFillNew.put(ServletRequest.class.getName(), fieldInfo);

                } else if (ServletResponse.class.isAssignableFrom(field.getType())) {

                    toFillNew.put(ServletResponse.class.getName(), fieldInfo);

                } else {

                    String parameterName = annotation.value();
                    if (StringUtils.isEmpty(parameterName)) {
                        parameterName = field.getName();
                    }

                    toFillNew.put(parameterName, fieldInfo);
                }
            }

            toAnalyze = toAnalyze.getSuperclass();
        }
        this.toFillFields = Cf.toHashMap(toFillNew);
    }

    protected T parseRequest(ServletRequest request, ServletResponse response) {
        T result;
        try {
            result = toFillConstructor.newInstance();
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }

        for (Map.Entry<String, FieldInfo> entry : toFillFields.entrySet()) {
            final String parameterName = entry.getKey();

            final Object value;
            if (ServletRequest.class.getName().equals(parameterName)) {
                value = request;
            } else if (ServletResponse.class.getName().equals(parameterName)) {
                value = response;
            } else {
                value = request.getParameter(parameterName);
            }

            if (value == null) {
                if (entry.getValue().required) {
                    throw new BadRequestException("No " + parameterName + " is specified");
                } else {
                    continue;
                }
            }

            final Field field = entry.getValue().field;
            final Class<?> fieldType = field.getType();

            try {
                if (boolean.class.equals(fieldType) || Boolean.class.equals(fieldType)) {
                    try {
                        Option<Integer> val = parseInteger(String.valueOf(value));
                        if (val.isPresent()) {
                            field.set(result, val.get() != 0);
                        } else {
                            field.set(result, Boolean.valueOf(String.valueOf(value)));
                        }
                    } catch (NumberFormatException exc) {
                        throw new BadRequestException("Incorrect '" + parameterName
                                + "' parameter value: '" + value + "'", exc);
                    }
                } else if (int.class.equals(fieldType) || Integer.class.equals(fieldType)) {
                    try {
                        field.set(result, Integer.valueOf(String.valueOf(value)));
                    } catch (NumberFormatException exc) {
                        throw new BadRequestException("Incorrect '" + parameterName
                                + "' parameter value: '" + value + "'", exc);
                    }
                } else if (long.class.equals(fieldType) || Long.class.equals(fieldType)) {
                    try {
                        field.set(result, Long.valueOf(String.valueOf(value)));
                    } catch (NumberFormatException exc) {
                        throw new BadRequestException("Incorrect '" + parameterName
                                + "' parameter value: '" + value + "'", exc);
                    }
                } else if (PassportUidOrZero.class.equals(fieldType)) {
                    try {
                        field.set(result,
                                PassportUidOrZero.fromUid(Long.parseLong(String.valueOf(value))));
                    } catch (Exception exc) {
                        throw new BadRequestException("Incorrect UID: '" + value + "'", exc);
                    }
                } else if (TargetType.class.equals(fieldType)) {
                    final TargetType targetType = TargetType
                            .getResolver()
                            .valueOfO(String.valueOf(value))
                            .getOrThrow(() -> new BadRequestException("Target type '" + value + "' is not supported"));
                    field.set(result, targetType);
                } else {
                    field.set(result, value);
                }
            } catch (IllegalAccessException exc) {
                throw ExceptionUtils.translate(exc);
            }
        }

        return result;
    }

    private Option<Integer> parseInteger(String value) {
        try {
            return Option.of(Integer.valueOf(value));
        } catch (NumberFormatException e) {
            return Option.empty();
        }
    }
}
