package ru.yandex.chemodan.app.docviewer.web.framework;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;

import org.dom4j.Attribute;
import org.dom4j.Comment;
import org.dom4j.Document;
import org.dom4j.DocumentType;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.Node;
import org.dom4j.Text;
import org.dom4j.tree.DefaultElement;
import org.dom4j.tree.NamespaceStack;

public class JSONWriter implements Closeable {
    protected static final JSONFormat DEFAULT_FORMAT = JSONFormat.BASIC_OUTPUT;

    /** The Writer used to output to */
    protected Writer writer;

    /** The Stack of namespaceStack written so far */
    private NamespaceStack namespaceStack = new NamespaceStack();

    /** The format used by this writer */
    private final JSONFormat format;

    /** Whether a flush should occur after writing a document */
    private boolean autoFlush;

    /** buffer used when escaping strings */
    private final StringBuilder buffer = new StringBuilder();

    public JSONWriter(OutputStream out) throws UnsupportedEncodingException {
        this.format = DEFAULT_FORMAT;
        this.writer = createWriter(out, format.getEncoding());
        this.autoFlush = true;
    }

    /**
     * Get an OutputStreamWriter, use preferred encoding.
     *
     */
    protected Writer createWriter(OutputStream outStream, String encoding)
            throws UnsupportedEncodingException {
        return new BufferedWriter(new OutputStreamWriter(outStream, encoding));
    }

    /**
     * Flushes the underlying Writer
     *
     * @throws IOException
     *             DOCUMENT ME!
     */
    public void flush() throws IOException {
        writer.flush();
    }

    /**
     * Closes the underlying Writer
     *
     */
    public void close() throws IOException {
        writer.close();
    }

    /**
     * <p>
     * This will print the <code>Document</code> to the current Writer.
     * </p>
     *
     * <p>
     * Warning: using your own Writer may cause the writer's preferred character
     * encoding to be ignored. If you use encodings other than UTF8, we
     * recommend using the method that takes an OutputStream instead.
     * </p>
     *
     * <p>
     * Note: as with all Writers, you may need to flush() yours after this
     * method returns.
     * </p>
     *
     * @param doc
     *            <code>Document</code> to format.
     *
     */
    public void write(Document doc) throws IOException {
        if (doc.getDocType() != null) {
            writeDocType(doc.getDocType());
            writer.write(" = ");
        }
        writer.write("{ ");

        for (int i = 0, size = doc.nodeCount(); i < size; i++) {
            if (i > 0) {
                writer.write(", ");
            }
            Node node = doc.node(i);
            writeNode(node);
        }

        writer.write(" ");
        writer.write("}");

        if (autoFlush) {
            flush();
        }
    }

    /**
     * <p>
     * Print out a {@link String}, Perfoms the necessary entity escaping and
     * whitespace stripping.
     * </p>
     *
     * @param text
     *            is the text to output
     *
     */
    public void write(String text) throws IOException {
        writeString(text);

        if (autoFlush) {
            flush();
        }
    }

    /**
     * Writes the given {@link Node}.
     *
     * @param node
     *            <code>Node</code> to output.
     *
     */
    public void write(Node node) throws IOException {
        writeNode(node);

        if (autoFlush) {
            flush();
        }
    }

    // Implementation methods
    // -------------------------------------------------------------------------
    protected void writeElement(Element element) throws IOException {
        writer.write("\"");
        writer.write(getJsonElementName(element));
        writer.write("\"");
        writer.write(": ");
        writeElementContent(element);
    }

    /**
     * @param element in document
     * @return qualified name of element massaged into a valid JavaScript
     * identifier name to make JSON output valid. We replace hyphens ("-")
     * with underscores ("_"). All other invalid characters (e.g., "+")
     * are replaced with Unicode code point sequence of the form "_uXXXX_"
     * (e.g., "_u002B_"). JavaScript reserved words (e.g., "protected")
     * are prefixed with underscores (e.g., "_protected").
     */
    private String getJsonElementName(Element element) {
        char[] block = null;
        int i;
        int last = 0;
        String name = element.getQualifiedName().replaceAll("-", "_");

        int size = name.length();
        for (i = 0; i < size; i++) {
            char c = name.charAt(i);

            if  (   (   (i == 0)
                    &&  (!Character.isJavaIdentifierStart(c)))
                    ||  (!Character.isJavaIdentifierPart(c))) {
                int codePoint = Character.codePointAt(name, i);
                if (block == null) {
                    block = name.toCharArray();
                }

                buffer.append(block, last, i - last);
                buffer.append(String.format("_u%04X_", codePoint));
                last = i + 1;
            }

        }

        if (last == 0) {
            return name;
        }

        if (last < size) {
            buffer.append(block, last, i - last);
        }

        String result = buffer.toString();
        buffer.setLength(0);

        return result;
    }

    protected void writeElementMixedContent(Element element) throws IOException {
        int attributeCount = element.attributeCount();
        writer.write("[ ");

        for (int i = 0; i < attributeCount; i++) {
            if (i > 0) {
                writer.write(", ");
            }
            writeAttribute(element.attribute(i));
        }
        for (int i = 0, nodesWritten = 0, nodeCount = element.nodeCount(); i < nodeCount; i++) {
            Node node = element.node(i);

            // Skip any whitespace-only Text nodes
            if  (   (node instanceof Text)
                    &&  (node.getText().trim().length() == 0)) {
                continue;
            }

            if ((attributeCount + nodesWritten) > 0) {
                writer.write(", ");
            }
            if  (   (node instanceof Element)
                    ||  (format.equals(JSONFormat.BADGER_FISH))) {
                writer.write("{ ");

                if (!(node instanceof Element)) {
                    writer.write("\"$\": ");
                }
                writeNode(node);

                writer.write(" ");

                writer.write("}");

            } else {
                writeNode(node);
            }

            nodesWritten++;
        }
        writer.write(" ");

        writer.write("]");
    }

    protected void writeElementContent(Element element) throws IOException {

        // Mixed content (element and text nodes) at the same level become
        // array elements.
        if (element.hasMixedContent()) {
            writeElementMixedContent(element);
            return;
        }

        // BASIC_OUTPUT & RABBIT_FISH: Text content goes directly in the value
        // of an object. For BADGER_FISH, we end up here processing the "$:"
        // property we add below.
        if  (   (element.attributeCount() == 0)
                &&  element.isTextOnly()
                &&  (   format.equals(JSONFormat.BASIC_OUTPUT)
                ||  format.equals(JSONFormat.RABBIT_FISH)
                ||  element.getName().equals("$"))) {
            writeNodeText(element);
            return;
        }

        // We have to collect all children with the same name into an array
        // which becomes the value of that property
        ArrayList<ArrayList<Node>> properties = new ArrayList<>();

        // Loop over the attributes to help determine answers to the above
        for (int i = 0, attributeCount = element.attributeCount(); i < attributeCount; i++) {
            addProperty(properties, element.attribute(i));
        }

        // Concatenate any Text nodes into a single property
        if (element.isTextOnly()) {
            addProperty(properties, new DefaultElement("$").addText(element.getText()));
        }

        // Loop over the nodes to help determine answers to the above
        for (int i = 0, nodeCount = element.nodeCount(); i < nodeCount; i++) {
            Node node = element.node(i);

            if (node instanceof Namespace) {
                // namespaces not supported yet

            } else if (node instanceof Comment) {
                // comments ignored

            } else if (node instanceof Element) {
                addProperty(properties, node);

            } else if (node instanceof Text) {
                // text nodes were concatenated above

            } else {
                // ignore everything else
            }
        }

        writer.write("{ ");

        for (int i = 0, propertyCount = properties.size(); i < propertyCount; i++) {
            if (i > 0) {
                writer.write(", ");
            }
            writeProperty(properties.get(i));
        }
        writer.write(" ");

        writer.write("}");
    }

    protected void addProperty( ArrayList<ArrayList<Node>> properties,
                                Node property) {
        ArrayList<Node> targetPropertyList = null;
        if  (   (property instanceof Element)
                &&  (!property.getName().equals("$"))) {
            for (ArrayList<Node> propertyList : properties) {
                if  (   (propertyList.get(0) instanceof Element)
                        &&  (propertyList.get(0).getName().equals(property.getName()))) {
                    targetPropertyList = propertyList;
                }
            }
        }
        if (targetPropertyList == null) {
            targetPropertyList = new ArrayList<>();
            properties.add(targetPropertyList);
        }
        targetPropertyList.add(property);
    }

    protected void writeProperty(ArrayList<Node> property) throws IOException {
        if (property.size() == 1) {
            writeNode(property.get(0));

        } else {
            String propertyName = property.get(0).getName();
            writer.write("\"");
            writer.write(propertyName);
            writer.write("\": ");
            writer.write("[ ");

            Iterator<Node> nodeIterator = property.iterator();
            for (int nodeIndex = 0; nodeIterator.hasNext(); nodeIndex++) {
                if (nodeIndex > 0) {
                    writer.write(", ");
                }
                writeElementContent((Element)(nodeIterator.next()));
            }

            writer.write(" ");
            writer.write("]");
        }
    }

    protected void writeDocType(DocumentType docType) throws IOException {
        if (docType != null) {
            docType.write(writer);
        }
    }

    protected void writeNamespace(Namespace namespace) throws IOException {
        if (namespace != null) {
            writeNamespace(namespace.getPrefix(), namespace.getURI());
        }
    }

    protected void writeAttribute(Attribute attribute) throws IOException {
        writer.write("\"");
        if  (   format.equals(JSONFormat.BADGER_FISH)
                ||  format.equals(JSONFormat.RABBIT_FISH)) {
            writer.write("@");
        }
        writer.write(attribute.getQualifiedName());
        writer.write("\": ");
        writeString(attribute.getText());
    }

    /**
     * Writes the SAX namepsaces
     *
     * @param prefix
     *            the prefix
     * @param uri
     *            the namespace uri
     *
     * @throws IOException
     */
    protected void writeNamespace(String prefix, String uri)
            throws IOException {
        throw new IOException("Namespaces not yet supported!");
    }

    protected void writeTextNode(Text textNode) throws IOException {
        writeNodeText(textNode);
    }

    protected void writeNodeText(Node node) throws IOException {
        writeString(node.getText());
    }

    protected void writeString(String text) throws IOException {
        if (text != null) {
            text = escapeElementEntities(text.trim());

            writer.write("\"");
            writer.write(text);
            writer.write("\"");
        }
    }

    protected void writeNode(Node node) throws IOException {
        int nodeType = node.getNodeType();

        switch (nodeType) {
            case Node.NAMESPACE_NODE:
                writeNamespace((Namespace) node);

                break;

            case Node.ATTRIBUTE_NODE:
                writeAttribute((Attribute) node);

                break;

            case Node.ELEMENT_NODE:
                writeElement((Element) node);

                break;

            case Node.TEXT_NODE:
                writeTextNode((Text) node);

                break;

            case Node.ENTITY_REFERENCE_NODE:
                // Skip entities, whatever they are

                break;

            case Node.COMMENT_NODE:
                // Skip comments

                break;

            case Node.DOCUMENT_NODE:
                write((Document) node);

                break;

            case Node.DOCUMENT_TYPE_NODE:
                writeDocType((DocumentType) node);

                break;

            default:
                throw new IOException("Invalid node type: " + node);
        }
    }

    /**
     * This will take the pre-defined entities in XML 1.0 and convert their
     * character representation to the appropriate entity reference, suitable
     * for XML attributes.
     *
     */
    protected String escapeElementEntities(String text) {
        char[] block = null;
        int i;
        int last = 0;
        int size = text.length();

        for (i = 0; i < size; i++) {
            String entity = null;
            char c = text.charAt(i);

            switch (c) {
                case '"':
                    entity = "\\\"";

                    break;

                case '/':
                    entity = "\\/";

                    break;

                case '\\':
                    entity = "\\\\";

                    break;
            }

            if (entity != null) {
                if (block == null) {
                    block = text.toCharArray();
                }

                buffer.append(block, last, i - last);
                buffer.append(entity);
                last = i + 1;
            }
        }

        if (last == 0) {
            return text;
        }

        if (last < size) {
            if (block == null) {
                block = text.toCharArray();
            }

            buffer.append(block, last, i - last);
        }

        String answer = buffer.toString();
        buffer.setLength(0);

        return answer;
    }

    /**
     * This will take the pre-defined entities in XML 1.0 and convert their
     * character representation to the appropriate entity reference, suitable
     * for XML attributes.
     *
     */
    protected String escapeAttributeEntities(String text) {
        char[] block = null;
        int i;
        int last = 0;
        int size = text.length();

        for (i = 0; i < size; i++) {
            String entity = null;
            char c = text.charAt(i);

            switch (c) {
                case '"':
                    entity = "\\\"";

                    break;

                case '/':
                    entity = "\\/";

                    break;

                case '\\':
                    entity = "\\\\";

                    break;
            }

            if (entity != null) {
                if (block == null) {
                    block = text.toCharArray();
                }

                buffer.append(block, last, i - last);
                buffer.append(entity);
                last = i + 1;
            }
        }

        if (last == 0) {
            return text;
        }

        if (last < size) {
            if (block == null) {
                block = text.toCharArray();
            }

            buffer.append(block, last, i - last);
        }

        String answer = buffer.toString();
        buffer.setLength(0);

        return answer;
    }

}
