package ru.yandex.chemodan.app.docviewer.web.framework;

import java.util.List;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.user.StoredUserDocument;
import ru.yandex.chemodan.app.docviewer.dao.user.UserDocumentsDao;
import ru.yandex.chemodan.app.docviewer.disk.DiskManager;
import ru.yandex.chemodan.app.docviewer.storages.FileLink;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetTypeHolder;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class PreviewHistoryManager {

    public static final Logger logger = LoggerFactory.getLogger(PreviewHistoryManager.class);

    private final DiskManager diskManager;
    private final UserDocumentsDao userDocumentsDao;
    private final FileStorage resultHolder;
    private final PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;

    public final DynamicProperty<Integer> resultsCleanupTtl =
            new DynamicProperty<>("docviewer.user-previews.cleanup.days.ttl", 180);

    public final DynamicProperty<Integer> maxDocumentPerUser =
            new DynamicProperty<>("docviewer.user-previews.max.count", -1);

    public Option<String> resolvePreview(ActualUri uri, PassportUidOrZero uid, String size) {
        Option<StoredUserDocument> document = userDocumentsDao.findDocument(uid, uri);
        if (!document.isPresent()) {
//            document = checkByUrl(uri, uid);
        }
        return document.map(sd -> {
            String contentType = pdfRenderTargetTypeHolder.getTargetType().getMimeType();
            String stid = MulcaId.fromMulcaUri(UrlUtils.uri(sd.getPreviewFileLink())).getStid();
            return diskManager.generateZaberunUrl(stid, uid, "preview", size, contentType);
        });
    }

    /**
     * Allows to search already converted preview among all users by uri
     * @param uri
     * @param uid
     * @return
     */
    private Option<StoredUserDocument> checkByUrl(ActualUri uri, PassportUidOrZero uid) {
        ListF<StoredUserDocument> documentByUrl = userDocumentsDao.findDocumentByUrl(uri);
        if (!documentByUrl.isEmpty()) {
            StoredUserDocument first = documentByUrl.first();
            StoredUserDocument sd = new StoredUserDocument(first, uid);
            userDocumentsDao.saveOrUpdateDocument(sd);
            return Option.of(sd);
        }
        return Option.empty();
    }

    public void addPreviewIfAbsent(PassportUidOrZero uid, ActualUri uri, File2 file, String type) {
        Option<StoredUserDocument> document = userDocumentsDao.findDocument(uid, uri);
        if (!document.isPresent()) {
            addPreview(uid, uri, file, type);
            cleanup(uid, Option.empty());
        }
        file.deleteRecursiveQuietly();
    }

    private void addPreview(PassportUidOrZero uid, ActualUri uri, File2 file, String type) {
        FileLink fileLink = resultHolder.put(file);
        StoredUserDocument doc = StoredUserDocument.builder().uid(uid).uri(uri).previewFileLink(fileLink).type(type).build();
        userDocumentsDao.saveOrUpdateDocument(doc);
    }

    public void delete(StoredUserDocument storedUserDocument) {
        FileLink fileLink = resultHolder.toFileLink(storedUserDocument.getPreviewFileLink());
        resultHolder.delete(fileLink);
        userDocumentsDao.delete(storedUserDocument);
        logger.info("Cleaned {}", storedUserDocument);
    }

    public void delete(PassportUidOrZero uid, ActualUri rewrite) {
        Option<StoredUserDocument> document = userDocumentsDao.findDocument(uid, rewrite);
        document.ifPresent(storedUserDocument -> delete(storedUserDocument));

    }

    public void cleanup(PassportUidOrZero uid, Option<Integer> count) {
        Integer cnt = count.getOrElse(maxDocumentPerUser.get());
        if (cnt > -1) {
            List<StoredUserDocument> documents = userDocumentsDao.findDocuments(uid, 100, cnt);
            documents.forEach(storedUserDocument -> delete(storedUserDocument));
        }
    }

    public void cleanup(Instant now, Option<Duration> age) {
        cleanup(now, age.getOrElse(Duration.standardDays(resultsCleanupTtl.get())));
    }

    public void cleanup(Instant now, Duration age) {
        userDocumentsDao.deleteByLastAccessLessBatch(now.minus(age), this::delete);
    }
}
