package ru.yandex.chemodan.app.docviewer.web.framework;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.dao.rights.UriRightsDao;
import ru.yandex.chemodan.app.docviewer.web.framework.exception.ForbiddenException;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class WebSecurityManager {

    private static final Logger logger = LoggerFactory.getLogger(WebSecurityManager.class);

    @Autowired
    private UriRightsDao uriRightsDao;
    @Autowired
    private UriHelper uriHelper;

    public void validateUrl(String url, PassportUidOrZero uid) {
        validateUrl(url, uid, Option.empty());
    }

    public void validateUrl(String url, PassportUidOrZero uid, Option<String> serpParams) {
        uriHelper.checkOriginalUrlAllowed(url, uid, serpParams);
    }

    public ActualUri validateRightAndGetActualUri(DocumentSourceInfo source) {
        ActualUri uri = uriHelper.rewrite(source);
        validateUrlRightUsingUid(source.getUid(), uri);
        return uri;
    }

    public void validateUrlRightUsingUid(PassportUidOrZero uid, ActualUri actualUri)
            throws ForbiddenException
    {
        if (uriRightsDao.findExistsUriRight(actualUri, uid)) {
        } else if (uid.isAuthenticated()
                && uriRightsDao.findExistsUriRight(actualUri, PassportUidOrZero.zero()))
        {
            logger.debug("Access granted for anyone");
        } else {
            logger.debug("Access denied UID '{}' actualUri '{}'", uid, actualUri);
            throw new ForbiddenException();
        }
    }

    public void validateFileRightUsingUid(PassportUidOrZero uid, String fileId)
            throws ForbiddenException
    {

        if (uriRightsDao.validate(fileId, uid)) {
        } else if (uid.isAuthenticated()
                && uriRightsDao.validate(fileId, PassportUidOrZero.zero()))
        {
            logger.debug("Access granted for anyone");
        } else {
            logger.debug("Access denied user '{}' file '{}'", uid, fileId);
            throw new ForbiddenException();
        }
    }

}
