package ru.yandex.chemodan.app.docviewer.adapters.poppler;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.Properties;

import javax.imageio.ImageIO;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.result.PageInfo;
import ru.yandex.chemodan.app.docviewer.convert.result.PagesInfo;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetType;
import ru.yandex.chemodan.boot.ChemodanPropertiesLoadStrategy;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.property.PropertiesHolder;
import ru.yandex.misc.property.load.PropertiesLoader;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.test.TestBase;
import ru.yandex.misc.version.SimpleAppName;

/**
 * @author ssytnik
 */
public class PopplerAdapterTest extends TestBase {

    private PopplerAdapter popplerAdapter;

    @Before
    public void init() {
        PropertiesLoader.initialize(new ChemodanPropertiesLoadStrategy(new SimpleAppName("docviewer", "web"), false));
        Properties props = PropertiesHolder.properties();
        popplerAdapter = new PopplerAdapter(
                props.getProperty("poppler.pdfinfo"),
                props.getProperty("poppler.pdftoppm"),
                props.getProperty("poppler.pdfseparate"),
                150);
    }

    @Test
    public void pageInfos() {
        FileUtils.withFile(
                new ClassPathResourceInputStreamSource(TestManager.class, "test/pdf/jean.pdf"),
                sourceFile -> {
                    Assert.equals(
                            new PageInfo(3, 389.8f, 510.8f),
                            popplerAdapter.getSinglePageInfo(sourceFile, 3));

                    Assert.equals(
                            Cf.list(new PageInfo(1, 385.834f, 607.502f), new PageInfo(2, 389.8f, 510.8f)),
                            popplerAdapter.getPagesInfo(sourceFile, 1, 2).getPageInfos());
                });
    }

    @Test
    public void parsePagesInfo() {
        Assert.equals(
                new PagesInfo(Cf.list(
                        new PageInfo(1, 875.906f, 314.646f),
                        new PageInfo(2, 428.031f, 314.646f),
                        new PageInfo(3, 428.031f, 314.646f))),
                popplerAdapter.parsePagesInfo(getPdfinfoOutput(false))
                );

        Assert.equals(
                new PagesInfo(Cf.list(
                        new PageInfo(1, 875.906f, 314.646f),
                        new PageInfo(2, 314.646f, 428.031f),
                        new PageInfo(3, 428.031f, 314.646f))),
                popplerAdapter.parsePagesInfo(getPdfinfoOutput(true))
                );
    }


    private String getPdfinfoOutput(boolean withRotationInfo) {
        return
                "Title:          01\n" +
                "Author:         S\n" +
                "Creator:        CorelDRAW X5\n" +
                "Producer:       Corel PDF Engine Version 15.2.0.661\n" +
                "CreationDate:   Tue Apr 19 10:40:35 2011\n" +
                "ModDate:        Tue Apr 19 11:19:14 2011\n" +
                "Tagged:         yes\n" +
                "Form:           AcroForm\n" +
                "Pages:          170\n" +
                "Encrypted:      no\n" +
                "Page    1 size: 875.906 x 314.646 pts\n" +
                (withRotationInfo ? "Page    1 rot:  0\n" : "") +
                "Page    2 size: 428.031 x 314.646 pts\n" +
                (withRotationInfo ? "Page    2 rot:  90\n" : "") +
                "Page    3 size: 428.031 x 314.646 pts\n" +
                (withRotationInfo ? "Page    3 rot:  0\n" : "") +
                "File size:      12370492 bytes\n" +
                "Optimized:      no\n" +
                "PDF version:    1.6\n";
    }

    @Test
    @Ignore
    public void testPdfPreviewSizes() {
        FileUtils.withTemporaryDirectory("test", "", (Function<File2, Void>) tmpDir -> {
            File2 source = File2.fromFileUrl(TestResources.Adobe_Acrobat_jean);
            File2 jpgTarget = tmpDir.child("test.jpg");
            File2 pngTarget = tmpDir.child("test.png");

            popplerAdapter.convertImpl(source, jpgTarget, 1,
                    ResizeOption.scale(DimensionO.cons(385, 607)), PdfRenderTargetType.JPG, true);
            popplerAdapter.convertImpl(source, pngTarget, 1,
                    ResizeOption.scale(DimensionO.cons(385, 607)), PdfRenderTargetType.PNG, true);

            System.out.println(jpgTarget.length());
            System.out.println(pngTarget.length());

            source = File2.fromFileUrl(TestResources.Adobe_Acrobat_1_5_114p);
            jpgTarget.deleteIfExists();
            pngTarget.deleteIfExists();

            popplerAdapter.convertImpl(source, jpgTarget, 1,
                    ResizeOption.scale(DimensionO.cons(595, 841)), PdfRenderTargetType.JPG, true);
            popplerAdapter.convertImpl(source, pngTarget, 1,
                    ResizeOption.scale(DimensionO.cons(595, 841)), PdfRenderTargetType.PNG, true);

            System.out.println(jpgTarget.length());
            System.out.println(pngTarget.length());

            return null;
        });
    }

    @Test
    public void renderWithScale() {
        renderAndCheckSize(ResizeOption.scale(DimensionO.cons(300, 400)), new Dimension(255, 400));
        renderAndCheckSize(ResizeOption.scale(DimensionO.cons(2000, 2000)), new Dimension(1271, 2000));
        renderAndCheckSize(ResizeOption.scale(DimensionO.consW(300)), new Dimension(300, 473));
        renderAndCheckSize(ResizeOption.scale(DimensionO.consH(400)), new Dimension(255, 400));
    }

    @Test
    public void renderWithBound() {
        renderAndCheckSize(ResizeOption.bound(DimensionO.cons(300, 400)), new Dimension(255, 400));
        renderAndCheckSize(ResizeOption.bound(DimensionO.cons(2000, 2000)), new Dimension(805, 1266));
    }

    private void renderAndCheckSize(final ResizeOption resize, final Dimension expected) {
        FileUtils.withEmptyTemporaryFile("test", "", target -> {
            File2 source = File2.fromFileUrl(TestResources.Adobe_Acrobat_jean);

            popplerAdapter.convertImpl(source, target, 1, resize, PdfRenderTargetType.PNG, true);

            try {
                BufferedImage img = ImageIO.read(target.getFile());
                Assert.equals(expected, new Dimension(img.getWidth(), img.getHeight()));
            } catch (IOException e) {
                throw IoUtils.translate(e);
            }
        });
    }
}
