package ru.yandex.chemodan.app.docviewer.convert;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.docviewer.DocviewerSpringTestBase;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUri;
import ru.yandex.chemodan.app.docviewer.dao.uris.StoredUriDao;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.State;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
public class ConvertManagerTest extends DocviewerSpringTestBase {

    @Autowired
    private TestManager testManager;

    @Autowired
    private StoredResultDao storedResultDao;

    @Autowired
    private StoredUriDao storedUriDao;

    @Autowired
    private ConvertManager convertManager;

    @Value("${convert.max.retries.count}")
    private int maxRetriesCount;

    private void checkConvertErrorType(String url, TargetType target, ErrorCode error) {
        StoredUri storedUri = storedUriDao.find(new ActualUri(url)).get();
        StoredResult result = storedResultDao.find(storedUri.getFileId().get(), target).get();
        Assert.equals(error, result.getErrorCode().get());
    }

    @Test
    public void testPreviewMimeTypes() {
        SetF<String> types = Cf.set("application/djvu", "image/x-djvu", "application/octet-stream");
        SetF<String> supported = convertManager.getPreviewMimeTypes(types);
        Assert.isTrue(supported.containsTs("application/djvu"));
        Assert.isTrue(supported.containsTs("image/x-djvu"));
        Assert.equals(supported.size(), 2);
    }

    @Test
    public void unknownCopyError() {
        String url = "http://asdfaasdfasdfasdfasdf.net";
        final State finalState = testManager.waitUriToComplete(
                PassportUidOrZero.zero(), url, TargetType.HTML_WITH_IMAGES);
        Assert.equals(State.COPY_ERROR, finalState);
        StoredUri storedUri = storedUriDao.find(new ActualUri(url)).get();
        Assert.equals(ErrorCode.UNKNOWN_COPY_ERROR, storedUri.getErrorCode().get());
    }

    @Test
    public void unknownConvertError() {
        String url = UriUtils.toUrlString(TestResources.BROKEN_JPG);
        final State finalState = testManager.waitUriToComplete(
                PassportUidOrZero.zero(), url, TargetType.HTML_WITH_IMAGES);
        Assert.equals(State.CONVERTING_ERROR, finalState);
        checkConvertErrorType(url, TargetType.HTML_WITH_IMAGES, ErrorCode.UNKNOWN_CONVERT_ERROR);
    }

    @Test
    public void tooManyAttemptsToRetry() {
        String url = UriUtils.toUrlString(TestResources.BROKEN_JPG);
        testManager.cleanupUri(PassportUidOrZero.zero(), url);

        for (int i = 0; i < maxRetriesCount; i++) {
            State finalState = testManager.waitUriToCompleteNoCleanup(
                    PassportUidOrZero.zero(), url, TargetType.HTML_WITH_IMAGES);
            Assert.equals(State.CONVERTING_ERROR, finalState);
            checkConvertErrorType(url, TargetType.HTML_WITH_IMAGES, ErrorCode.UNKNOWN_CONVERT_ERROR);
        }

        State finalState = testManager.waitUriToCompleteNoCleanup(
                PassportUidOrZero.zero(), url, TargetType.HTML_WITH_IMAGES);
        Assert.equals(State.CONVERTING_ERROR, finalState);
        checkConvertErrorType(url, TargetType.HTML_WITH_IMAGES, ErrorCode.TOO_MANY_RETRIES);
    }

    @Test
    public void doNotIncreaseFailedAttemptsCountForFileIsPasswordProtectedError() {
        String url = UriUtils.toUrlString(TestResources.SEVENZIP_WITH_PASSWORD);
        testManager.cleanupUri(PassportUidOrZero.zero(), url);

        State finalState = testManager.waitUriToCompleteNoCleanup(
                PassportUidOrZero.zero(), url, TargetType.HTML_WITH_IMAGES);

        Assert.equals(State.CONVERTING_ERROR, finalState);
        checkConvertErrorType(url, TargetType.HTML_WITH_IMAGES, ErrorCode.FILE_IS_PASSWORD_PROTECTED);

        StoredUri storedUri = storedUriDao.find(new ActualUri(url)).get();
        StoredResult result = storedResultDao.find(storedUri.getFileId().get(), TargetType.HTML_WITH_IMAGES).get();
        Assert.equals(0, result.getFailedAttemptsCount().get());
    }


}
