package ru.yandex.chemodan.app.docviewer.convert;

import java.net.URL;

import org.dom4j.Document;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.docviewer.DocviewerSpringTestBase;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.pdf.PdfToHtml;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.states.PagesInfoHelper;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfHelper;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetTypeHolder;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * This class is used to debug document processing locally
 *
 * @author akirakozov
 * @author ssytnik
 */
public class ConvertManagerTestTool extends DocviewerSpringTestBase {
    private static final String RESULT_PREFIX = "result.";

    @Autowired
    private FileStorage resultHolder;
    @Autowired
    private TestManager testManager;
    @Autowired
    private StoredResultDao storedResultDao;
    @Autowired
    private PdfToHtml pdfToHtml;
    @Autowired
    private PdfHelper pdfHelper;
    @Autowired
    private PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;


    @Test
    public void convertFile() {
        String pathToLocalFile = "test/jpg/JPEG.jpg";
        TargetType targetType = TargetType.HTML_WITH_IMAGES;
        convertFileAndSaveResultLocally(pathToLocalFile, targetType);
    }

    @Test
    public void convertAndProcessPdf() {
        String pathToLocalFile = "test/pdf/welcome.pdf";
        TargetType targetType = TargetType.HTML_WITH_IMAGES;
        int page = 1;

        Tuple2<String, String> result = convertFileAndSaveResultLocally(pathToLocalFile, targetType);
        savePdfBackground(result.get1(), page);
        processPdfToHtml(result.get2(), page);
    }

    private void savePdfBackground(String fileId, int page) {
        pdfHelper.removeHtmlBackground(fileId, page, DimensionO.WIDTH_900);
        File2 temporaryFile = pdfHelper.getHtmlBackgroundImageInplace(fileId, page, DimensionO.WIDTH_900, TargetType.HTML_WITH_IMAGES);
        String imageExtension = pdfRenderTargetTypeHolder.getTargetType().value();
        temporaryFile.readTo(new File2(RESULT_PREFIX + imageExtension));
    }

    private void processPdfToHtml(String fileName, int page) {
        Document resultHtml = PdfUtils.withExistingDocument(new File2(fileName), true, pdDocument -> {
            return pdfToHtml.getPageHtml(
                    PagesInfoHelper.toPagesInfo(pdDocument), Option.of(page), true, DimensionO.WIDTH_900);
        });
        Dom4jUtils.write(resultHtml, new FileOutputStreamSource(new File2(RESULT_PREFIX + "xml")));
    }



    private String convertAndGetId(URL url, TargetType targetType) {
        return testManager.makeAvailable(PassportUidOrZero.zero(), url, targetType);
    }

    private String convertAndGetId(String pathToFile, TargetType targetType) {
        return convertAndGetId(TestResources.class.getResource(pathToFile), targetType);
    }

    private String saveResultLocally(String fileId, TargetType targetType) {
        StoredResult storedResult = storedResultDao.find(fileId, targetType).get();
        InputStreamSource iss = resultHolder.get(resultHolder.toFileLink(storedResult.getFileLink().get()));
        String resultFileName = RESULT_PREFIX + storedResult.getConvertResultType().get().getExtension();
        iss.readTo(new File2(resultFileName));
        return resultFileName;
    }

    private Tuple2<String, String> convertFileAndSaveResultLocally(String pathToLocalFile, TargetType targetType) {
        String fileId = convertAndGetId(pathToLocalFile, targetType);
        String resultFileName = saveResultLocally(fileId, targetType);
        return Tuple2.tuple(fileId, resultFileName);
    }

}
