package ru.yandex.chemodan.app.docviewer.convert;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.dom4j.Document;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.AbstractSpringAwareTest;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.convert.pdf.PdfToHtml;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResult;
import ru.yandex.chemodan.app.docviewer.dao.results.StoredResultDao;
import ru.yandex.chemodan.app.docviewer.storages.FileStorage;
import ru.yandex.chemodan.app.docviewer.utils.ByteArrayOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlPostprocessor;
import ru.yandex.chemodan.app.docviewer.utils.html.HtmlSerializer;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfHelper;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileInputStreamSource;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;

@RunWith(value = Parameterized.class)
public class ConvertTestSuite extends AbstractSpringAwareTest {

    private static final Logger logger = LoggerFactory.getLogger(ConvertTestSuite.class);

    private static final String[] extensions = { "ai", "csv", "doc", "docx", "odp", "pdf", "pptx", "txt", "xls", "xml" };

    // XXX support pdf for texts, try to enable line wrapping in imagemagick
    private static boolean runPdfTests(String extension) {
        return ! "txt".equals(extension);
    }


    public static final String PATH_TEST_RESULTS_PREFIX = "target/test-results/suites/";

    @Autowired
    private TestManager testManager;

    @BeforeClass
    public static void beforeClass() {
        for (String extension : extensions) {
            final String path = PATH_TEST_RESULTS_PREFIX + extension;
            new File2(path).deleteRecursive();
            new File2(path).mkdirs();
        }
    }

    @Parameters
    public static Collection<Object[]> data() {
        List<Object[]> data = new ArrayList<>();

        for (String extension : extensions) {
            URL suiteFileUrl = ConvertTestSuite.class.getResource("/ru/yandex/chemodan/app/docviewer/test/" + extension
                    + "/suite/suite");
            File2 suiteFile = UriUtils.toFile(suiteFileUrl);
            File2 parentDir = suiteFile.parent();
            final List<String> list = parentDir.listNames().sorted();
            for (String file : list) {
                if (!file.endsWith("." + extension))
                    continue;
                data.add(new Object[] { file });
            }
        }
        return data;
    }

    @Autowired
    private ConvertManager convertManager;

    private final String extension;

    @Autowired
    private HtmlPostprocessor htmlPostprocessor;

    @Autowired
    private HtmlSerializer htmlSerializer;

    @Autowired
    private MimeDetector mimeDetector;

    @Autowired
    private PdfHelper pdfHelper;

    @Autowired
    private PdfToHtml pdfToHtml;

    @Autowired
    @Qualifier("resultHolder")
    private FileStorage resultHolder;

    private final File2 sourceFile;

    @Autowired
    private StoredResultDao storedResultDao;

    public ConvertTestSuite(String sourceFileName) {
        this.extension = StringUtils.substringAfterLast(sourceFileName, ".");

        URL suiteFileUrl = ConvertTestSuite.class.getResource("/ru/yandex/chemodan/app/docviewer/test/" + extension
                + "/suite/suite");
        File2 suiteFile = UriUtils.toFile(suiteFileUrl);
        File2 parentDir = new File2(suiteFile.getFile().getParentFile());

        this.sourceFile = new File2(parentDir, sourceFileName);
    }

    @Test
    public void testHtmlOnly() {
        final File resultsDirectory = new File(PATH_TEST_RESULTS_PREFIX + extension);

        final File2 inputFile = new File2(resultsDirectory, sourceFile.getName());
        sourceFile.readTo(inputFile);
        final String extensionContentType = mimeDetector.getMimeTypeByFilename(sourceFile
                .getAbsolutePath());

        FileUtils.withEmptyTemporaryFile("result-html", ".bin", file -> {

            logger.info("Converting file '{}' to '{}'...", new Object[] { inputFile, file });

            ConvertResultInfo convertResultInfo = convertManager.convert(
                    inputFile,
                    mimeDetector.getMimeType(inputFile,
                            Collections.singleton(extensionContentType)), TargetType.HTML_ONLY,
                    new FileOutputStreamSource(file));

            if (convertResultInfo.isPdf()) {
                File2 resultFile = new File2(inputFile.parent(), inputFile.getName()
                        + ".nohtml.pdf");
                file.readTo(resultFile);
                logger.debug("File '{}' converted to '{}'", new Object[] { inputFile,
                        resultFile });

                final Document html = pdfToHtml.getPageHtml(convertResultInfo.getPagesInfo().get(), Option.empty(),
                        false, DimensionO.WIDTH_1024);

                new FileOutputStreamSource(new File2(inputFile.parent(), inputFile.getName()
                        + ".nohtml.html")).write(htmlSerializer.serializeToHtmlHandler(html));

            } else if (convertResultInfo.isZippedHtml()) {
                File2 resultFile = new File2(inputFile.parent(), inputFile.getName() + ".zip");
                file.readTo(resultFile);
                logger.debug("File '{}' converted to '{}'", new Object[] { inputFile,
                        resultFile });
            } else {
                throw new RuntimeException("Unsupported convert result type: " + convertResultInfo.getType());
            }
        });
    }

    @Test
    public void testHtmlWithImages() {
        final File2 resultsDirectory = new File2(PATH_TEST_RESULTS_PREFIX + extension);

        final File2 inputFile = new File2(resultsDirectory, sourceFile.getName());
        sourceFile.readTo(inputFile);

        String fileId = testManager.makeAvailable(PassportUidOrZero.zero(), UriUtils.toUrlString(sourceFile), TargetType.HTML_WITH_IMAGES);
        StoredResult storedResult = storedResultDao.find(fileId, TargetType.HTML_WITH_IMAGES).get();
        InputStreamSource result = resultHolder.get(resultHolder.toFileLink(storedResult
                .getFileLink().get()));

        File2 target;
        if (storedResult.isConvertResultTypePdf()) {
            File2 dir = new File2(inputFile.parent(), StringUtils.substringBeforeLast(
                    sourceFile.getName(), "."));
            dir.mkdirs();

            final Document html = pdfToHtml.getPageHtml(storedResult.getPagesInfo().get(), Option.empty(), true,
                    DimensionO.WIDTH_1024);
            htmlPostprocessor.preprocessOutput(html, Option.of(fileId), DimensionO.WIDTH_1024);

            ByteArrayOutputStreamSource temp = new ByteArrayOutputStreamSource();
            temp.write(htmlSerializer.serializeToHtmlHandler(html));
            String htmlString = new String(temp.getByteArray());
            htmlString = StringUtils.replace(htmlString, "('./htmlimage?id=" + fileId
                    + "&width=1024&name=bg-", "('bg-");
            new FileOutputStreamSource(new File2(dir, "withimages.html")).write(htmlString);

            for (int i = 0; i < Math.min(5, storedResult.getPages().get()); i++) {
                File2 bg = new File2(dir, "bg-" + i + ".png");
                pdfHelper.getHtmlBackgroundImageInplace(fileId, i + 1, DimensionO.WIDTH_1024, TargetType.HTML_WITH_IMAGES).readTo(bg);
            }

            PdfUtils.withExistingDocument(
                    result,
                    true,
                    PdfUtils.filterTextOperatorsHandler(PdfUtils.preserveInBackgroundHandler(true))
                            .asFunctionReturnParam()
                            .andThen(
                                    PdfUtils.writeHandler(new FileOutputStreamSource(new File2(
                                            inputFile.parent(), inputFile.getName()
                                                    + ".background.pdf")))));

            target = new File2(inputFile.parent(), inputFile.getName() + ".withimages.pdf");

        } else if (storedResult.isConvertResultTypeZippedHtml()) {
            target = new File2(inputFile.parent(), inputFile.getName() + "-2.zip");

        } else {
            throw new RuntimeException("Unsupported convert result type: " + storedResult.getConvertResultType());
        }

        result.readTo(target);
        logger.debug("File '{}' converted to '{}'", new Object[] { inputFile, target });
    }

    @Test
    public void testPdf() {
        if (!runPdfTests(extension)) {
            return;
        }

        final File resultsDirectory = new File(PATH_TEST_RESULTS_PREFIX + extension);

        File inputFile = new File(resultsDirectory, sourceFile.getName());
        sourceFile.readTo(inputFile);
        final FileInputStreamSource source = new FileInputStreamSource(inputFile);
        final String extensionContentType = mimeDetector.getMimeTypeByFilename(sourceFile
                .getAbsolutePath());

        File outputZipFile = new File(inputFile.getParentFile(), inputFile.getName() + ".pdf");

        logger.info("Converting file '{}' to '{}'...", new Object[] { inputFile, outputZipFile });

        convertManager.convert(source,
                mimeDetector.getMimeType(source, Collections.singleton(extensionContentType)),
                TargetType.PDF, new FileOutputStreamSource(outputZipFile));

        logger.debug("File '{}' converted to '{}'", new Object[] { inputFile, outputZipFile });
    }

    @Test
    @Ignore
    public void testPreview() {
        if (!runPdfTests(extension)) {
            return;
        }

        logger.info("Processing: {}, extension: {}", sourceFile, extension);

        final File2 resultsDirectory = new File2(PATH_TEST_RESULTS_PREFIX + extension);

        File2 inputFile = new File2(resultsDirectory, sourceFile.getName());
        sourceFile.readTo(inputFile);

        String previewurl = ApacheHttpClientUtils
                .downloadString("http://localhost:32405/previewurl?uid=0&width=1024&url="
                        + UrlUtils.urlEncode(UriUtils.toUrlString(sourceFile)));

        byte[] preview = ApacheHttpClientUtils.download("http://localhost:32400"
                + StringUtils.trimToEmpty(previewurl));
        File2 outputFile = new File2(inputFile.parent(), inputFile.getName() + ".png");
        new FileOutputStreamSource(outputFile).write(preview);
    }

    @Test
    public void testTextOnly() {
        final File resultsDirectory = new File(PATH_TEST_RESULTS_PREFIX + extension);

        File2 inputFile = new File2(resultsDirectory, sourceFile.getName());
        sourceFile.readTo(inputFile);
        final String extensionContentType = mimeDetector.getMimeTypeByFilename(sourceFile
                .getAbsolutePath());

        File2 outputZipFile = new File2(inputFile.parent(), inputFile.getName() + ".txt");

        logger.info("Converting file '{}' to '{}'...", new Object[] { inputFile, outputZipFile });

        convertManager.convert(inputFile,
                mimeDetector.getMimeType(inputFile, Collections.singleton(extensionContentType)),
                TargetType.PLAIN_TEXT, new FileOutputStreamSource(outputZipFile));

        logger.debug("File '{}' converted to '{}'", new Object[] { inputFile, outputZipFile });
    }

}
