package ru.yandex.chemodan.app.docviewer.convert.doc;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.w3c.dom.Document;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.AbstractSpringAwareTest;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.test.Assert;

@RunWith(value = Parameterized.class)
public class PoiExamplesTestSuite extends AbstractSpringAwareTest {

    private static final Logger logger = LoggerFactory.getLogger(PoiExamplesTestSuite.class);

    public static final String PATH_TEST_RESULTS = "target/test-results/poi-examples/doc";

    private static final File resultsDirectory = new File(PATH_TEST_RESULTS);

    @BeforeClass
    public static void beforeClass() {
        new File2(resultsDirectory).deleteRecursive();
        resultsDirectory.mkdirs();

        FileUtils.withZipFile(new UrlInputStreamSource(TestResources.Apache_Poi_Examples_Documents),
                zipFile -> {
                    for (Enumeration<? extends ZipEntry> enumeration = zipFile.entries(); enumeration
                            .hasMoreElements();)
                    {
                        ZipEntry zipEntry = enumeration.nextElement();

                        File2 sourceDoc = new File2(resultsDirectory, zipEntry.getName());
                        new ZipEntryInputStreamSource(zipFile, zipEntry).readTo(sourceDoc);

                        logger.debug("File '{}' extracted", zipEntry.getName());
                    }
                });
    }

    @Parameters
    public static Collection<Object[]> data() {
        return FileUtils.withZipFile(new UrlInputStreamSource(
                TestResources.Apache_Poi_Examples_Documents),
                zipFile -> {
                    List<String> names = new ArrayList<>();
                    for (Enumeration<? extends ZipEntry> enumeration = zipFile.entries(); enumeration
                            .hasMoreElements();)
                    {
                        ZipEntry zipEntry = enumeration.nextElement();
                        names.add(zipEntry.getName());
                    }

                    Collections.sort(names);
                    final List<Object[]> data = new ArrayList<>();
                    for (String name : names) {
                        data.add(new Object[] { name });
                    }
                    return data;
                });
    }

    @Autowired
    private DocConverter docConverter;

    private final DocumentBuilderFactory documentBuilderFactory;

    private final String entryName;

    @Autowired
    private MimeDetector mimeDetector;

    public PoiExamplesTestSuite(String entryName) {
        this.entryName = entryName;

        this.documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true);
    }

    @Test
    public void testDetectMime() {
        File2 inputFile = new File2(resultsDirectory, entryName);
        String mime = mimeDetector.getMimeType(inputFile);

        Assert.A.equals(MimeTypes.MIME_MICROSOFT_WORD, mime);
    }

    @Test
    public void testHtmlOnly() throws Exception {
        File2 inputFile = new File2(resultsDirectory, entryName);

        Document doc = documentBuilderFactory.newDocumentBuilder().newDocument();
        File2 outputZipFile = new File2(inputFile.parent(), inputFile.getName() + ".zip");

        logger.info("Converting file '{}' to '{}'...", new Object[] { inputFile, outputZipFile });

        docConverter.doConvert(inputFile, TargetType.HTML_ONLY, doc,
                new FileOutputStreamSource(outputZipFile), Option.empty());

        logger.debug("File '{}' converted to '{}'", new Object[] { inputFile, outputZipFile });
    }

//    pdf is not supported now
    @Test
    @Ignore
    public void testPdf() throws Exception {
        File2 inputFile = new File2(resultsDirectory, entryName);

        Document doc = documentBuilderFactory.newDocumentBuilder().newDocument();
        File2 outputPdfFile = new File2(inputFile.parent(), inputFile.getName() + ".pdf");

        logger.info("Converting file '{}' to '{}'...", new Object[] { inputFile, outputPdfFile });

        try {
            docConverter.doConvert(inputFile, TargetType.PDF, doc,
                    new FileOutputStreamSource(outputPdfFile), Option.empty());

            logger.debug("File '{}' converted to '{}'", new Object[] { inputFile, outputPdfFile });
        } finally {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.transform(new DOMSource(doc), new StreamResult(new File2(
                    inputFile.parent(), inputFile.getName() + ".xml").getFile()));
        }
    }
}
