package ru.yandex.chemodan.app.docviewer.convert.openoffice;

import java.awt.image.RenderedImage;
import java.io.IOException;
import java.net.URL;

import org.apache.pdfbox.util.PDFTextStripper;
import org.junit.Ignore;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.convert.result.ConvertResultInfo;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ImageUtils;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.web.DocviewerWebSpringTestBase;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.devnull.DevNullOutputStreamSource;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.test.Assert;

public class OpenOfficeConverterTest extends DocviewerWebSpringTestBase {

    @Autowired
    private TestManager testManager;

    @Autowired
    private OpenOfficeConverter openOfficeConverter;

    @Autowired
    private ConvertManager convertManager;


    @Test
    public void convertDocxToHtml() {
        String fileId = testManager.makeAvailable(PassportUidOrZero.zero(),
                UriUtils.toUrlString(TestResources.Microsoft_Word_12_001p), TargetType.HTML_ONLY);

        String htmlPageInfo = ApacheHttpClientUtils
                .downloadString("http://localhost:32405/htmlonlypageinfo?uid=0&id=" + fileId
                        + "&page=1");
        Assert.assertContains(htmlPageInfo, "67a43feb-1c53-4b31-887b-c7f0471708d8");

        String htmlInfo = ApacheHttpClientUtils.downloadString("http://localhost:32405/htmlonlyinfo?uid=0&id="
                + fileId + "&page=1");
        Assert.assertContains(htmlInfo, "67a43feb-1c53-4b31-887b-c7f0471708d8");
    }

    @Test
    public void convertDocxToHtmlWithImages() {
        convertDocxToHtml(TargetType.HTML_WITH_IMAGES, false);
    }

    @Test
    public void convertDocxToHtmlWithImagesForMobile() {
        convertDocxToHtml(TargetType.HTML_WITH_IMAGES_FOR_MOBILE, true);
    }

    private void convertDocxToHtml(TargetType type, boolean mobile) {
        String fileId = testManager.makeAvailable(PassportUidOrZero.zero(),
                UriUtils.toUrlString(TestResources.Microsoft_Word_12_001p_with_image), type);

        String htmlPageInfo = ApacheHttpClientUtils
                .downloadString("http://localhost:32405/htmlwithimagespageinfo?uid=0&id=" + fileId
                        + "&page=1" + (mobile ? "&mobile=true" : ""));
        Assert.assertContains(htmlPageInfo, "67a43feb-1c53-4b31-887b-c7f0471708d8");

        String htmlInfo = ApacheHttpClientUtils
                .downloadString("http://localhost:32405/htmlwithimagesinfo?uid=0&id=" + fileId
                        + "&page=1" + (mobile ? "&mobile=true" : ""));
        Assert.assertContains(htmlInfo, "67a43feb-1c53-4b31-887b-c7f0471708d8");

        checkHtmlImage(htmlPageInfo, mobile);
    }

    private void checkHtmlImage(String htmlPageInfo, boolean mobile) {
        int beginIndex = htmlPageInfo.indexOf("./htmlimage");
        int endIndex = htmlPageInfo.indexOf("\"", beginIndex);
        String imgUrl = htmlPageInfo.substring(beginIndex, endIndex).replaceAll("&amp;", "&") + "&uid=0" + (mobile ? "&mobile=true" : "");
        byte[] firstPageImage = ApacheHttpClientUtils.download("http://localhost:32405" + imgUrl);
        RenderedImage image = ImageUtils.read(new ByteArrayInputStreamSource(firstPageImage));
        Assert.assertEquals(738, image.getWidth());
        Assert.assertEquals(720, image.getHeight());
    }

    @Test
    public void convertExcelWithMacrossesToHtmlWithImages() {
        convertSpreadsheetWithMacrossesToHtmlWithImages(TestResources.Microsoft_Excel_binary_with_macros, this::macroChecker);
    }

    @Test
    public void convertOdsWithMacrossesToHtmlWithImages() {
        convertSpreadsheetWithMacrossesToHtmlWithImages(TestResources.OpenOffice_Calc_WithMacro, this::macroChecker);
    }

    @Test
    public void convertXlsmToHtmlWithImages() {
        convertSpreadsheetWithMacrossesToHtmlWithImages(TestResources.Microsoft_Excel_xlsm, this::textChecker);
    }

    @Test
    public void convertXltmToHtmlWithImages() {
        convertSpreadsheetWithMacrossesToHtmlWithImages(TestResources.Microsoft_Excel_xltm, this::textChecker);
    }

    private void convertSpreadsheetWithMacrossesToHtmlWithImages(URL resource, Function1V<String> checker) {
        String fileId = testManager.makeAvailable(PassportUidOrZero.zero(),
                UriUtils.toUrlString(resource), TargetType.HTML_WITH_IMAGES);

        String htmlPageInfo = ApacheHttpClientUtils
                .downloadString("http://localhost:32405/htmlwithimagespageinfo?uid=0&id=" + fileId + "&page=1");
        checker.accept(htmlPageInfo);
    }

    private void macroChecker(String htmlPageInfo) {
        Assert.assertFalse(htmlPageInfo.contains("macro"));
    }

    private void textChecker(String htmlPageInfo) {
        Assert.assertContains(htmlPageInfo, "findthisone");
        Assert.assertContains(htmlPageInfo, "superword");
    }

    @Test
    public void convertTxtToPdf() {
        final String text = "Привет текст";
        final ByteArrayInputStreamSource source = new ByteArrayInputStreamSource(text.getBytes());
        FileUtils.withEmptyTemporaryFile("tmp", ".pdf", result -> {
            openOfficeConverter.convert(source, MimeTypes.MIME_TEXT_PLAIN,
                    TargetType.PREVIEW, result.asOutputStreamTool(),
                    Option.empty());

            PdfUtils.withExistingDocument(result.asInputStreamTool(), true, d -> {
                PDFTextStripper stripper;
                try {
                    stripper = new PDFTextStripper("utf-8");
                    stripper.setForceParsing(true);
                    stripper.setSortByPosition(true);
                    stripper.setShouldSeparateByBeads(true);

//                    Assert.equals(text + "\n", stripper.getText(d));
                } catch (IOException e) {
                    IoUtils.translate(e);
                }
            });
        });
    }

    // XXX while this test fails, passwords for open office converter cannot be supported,
    // in spite of the fact that it accepts and understands them quite well, because there
    // is no way to recognize this situation:
    // ExtendedOpenOfficeDocumentConverter#loadAndExport(...) returns null, but not exception
    @Ignore // can we solve, or ignore?
    @Test
    public void convertOdpPasswordException() {
        try {
            convertManager.convertCommon(new UrlInputStreamSource(
                    TestResources.Microsoft_Word_97_001p_password), MimeTypes.MIME_MICROSOFT_WORD,
                    openOfficeConverter, TargetType.HTML_WITH_IMAGES, new DevNullOutputStreamSource());
            Assert.A.fail("UserException expected");
        } catch (UserException exc) {
            Assert.A.equals(ErrorCode.FILE_IS_PASSWORD_PROTECTED, exc.getErrorCode());
        }
    }

    @Test
    public void convertOdpPasswordExtract() {
        convertManager.convertCommon(new UrlInputStreamSource(
                TestResources.Microsoft_Word_97_001p_password), MimeTypes.MIME_MICROSOFT_WORD,
                openOfficeConverter, TargetType.HTML_WITH_IMAGES, new DevNullOutputStreamSource(),
                Option.of("password"));
    }

    @Test
    public void convertMultiplePageOdpTest() {
        ConvertResultInfo result =
                convertManager.convertCommon(new UrlInputStreamSource(
                        TestResources.OpenOffice_Multipage_odp), MimeTypes.MIME_OPENDOCUMENT_PRESENTATION,
                        openOfficeConverter, TargetType.PDF, new DevNullOutputStreamSource());
        Assert.equals(7, result.getPages());
    }

    @Test
    public void convertPptxTest() {
        ConvertResultInfo result =
                convertManager.convertCommon(new UrlInputStreamSource(
                                TestResources.Microsoft_PowerPoint_Docviewer_933), MimeTypes.MIME_OPENDOCUMENT_PRESENTATION,
                        openOfficeConverter, TargetType.PDF, new DevNullOutputStreamSource());

        Assert.equals(7, result.getPages());
    }

    @Test
    public void convertPptTest() {
        ConvertResultInfo result =
                convertManager.convertCommon(new UrlInputStreamSource(
                                TestResources.Microsoft_PowerPoint_97_003p), MimeTypes.MIME_OPENDOCUMENT_PRESENTATION,
                        openOfficeConverter, TargetType.PDF, new DevNullOutputStreamSource());

        Assert.equals(3, result.getPages());
    }
}
