package ru.yandex.chemodan.app.docviewer.convert.openoffice;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.app.docviewer.AbstractSpringAwareTest;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.devnull.DevNullOutputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.test.Assert;

@RunWith(value = Parameterized.class)
public class OpenOfficeConverterTestSuite extends AbstractSpringAwareTest {

    @Parameters
    public static Collection<Object[]> data() {
        List<Object[]> data = new ArrayList<>();
        for (TargetType targetType : new TargetType[]{TargetType.HTML_ONLY,
                TargetType.HTML_WITH_IMAGES, TargetType.PDF}) {
            for (URL url : new URL[]{ //
                    //
                    TestResources.Microsoft_Excel_97_001p, //
                    TestResources.Microsoft_Excel_97_001p_bug, //
//                    TestResources.Microsoft_Excel_97_177p, //
//                    TestResources.Microsoft_Excel_97_565p, //
//                    TestResources.Microsoft_Excel_97_582p, //
                    TestResources.Microsoft_RTF, //
                    TestResources.Microsoft_Word_12_001p, //
                    TestResources.Microsoft_Word_12_017p, //
                    TestResources.Microsoft_Word_95_001p, //
                    TestResources.Microsoft_Word_95_001p, //
                    TestResources.Microsoft_Word_97_001p, //
                    TestResources.Microsoft_Word_97_002p, //
                    TestResources.Microsoft_Word_97_025p, //
                    TestResources.Microsoft_Word_97_046p, //
                    TestResources.Microsoft_Word_97_050p, //
                    TestResources.OpenOffice_Calc_3_0_001p, //
                    TestResources.OpenOffice_Writer_3_2_001p, //
            }) {
                data.add(new Object[]{url, targetType, null});
            }
        }

        for (TargetType targetType : new TargetType[]{TargetType.PDF}) {
            for (URL url : new URL[]{ //
                    //
                    TestResources.Microsoft_PowerPoint_97_001p, //
                    TestResources.Microsoft_Word_97_102p, //
                    TestResources.OpenOffice_Impress_3_3_001p, //
                    //
            }) {
                data.add(new Object[]{url, targetType, null});
            }
        }

        for (URL url : new URL[]{ //
                //
                TestResources.Microsoft_RTF, //
                TestResources.Microsoft_Word_12_001p, //
                TestResources.Microsoft_Word_12_017p, //
                TestResources.Microsoft_Word_95_001p, //
                TestResources.Microsoft_Word_97_046p, //
                TestResources.Microsoft_Word_97_050p, //
                TestResources.Microsoft_Word_97_102p, //
                TestResources.OpenOffice_Writer_3_2_001p, //
                //
        }) {
            data.add(new Object[]{url, TargetType.PLAIN_TEXT, null});
        }

        for (TargetType targetType : new TargetType[]{TargetType.HTML_ONLY, TargetType.PDF,
                TargetType.PLAIN_TEXT}) {
            for (URL url : new URL[]{ //
                    //
//                    TestResources.Microsoft_Word_97_001p_password, //
                    //
            }) {
                data.add(new Object[]{url, targetType, ErrorCode.FILE_IS_PASSWORD_PROTECTED});
            }
        }

        return data;
    }

    @Autowired
    private ConvertManager convertManager;

    private final TargetType convertTargetType;

    private final ErrorCode errorCode;

    @Autowired
    private MimeDetector mimeDetector;

    @Autowired
    private OpenOfficeConverter openOfficeConverter;

    private final URL url;

    public OpenOfficeConverterTestSuite(URL url, TargetType convertTargetType, ErrorCode errorCode)
    {
        this.url = url;
        this.convertTargetType = convertTargetType;
        this.errorCode = errorCode;
    }

    @Test
    public void test() {
        Validate.notNull(url, "url is null");

        final InputStreamSource source = new UrlInputStreamSource(url);
        final String mimeType = mimeDetector.getMimeType(source);

        try {
            convertManager.convertCommon(source, mimeType, openOfficeConverter, convertTargetType,
                    new DevNullOutputStreamSource());

            if (errorCode != null)
                Assert.A.fail("UserException expected");
        } catch (UserException exc) {
            if (errorCode != null)
                Assert.A.equals(errorCode, exc.getErrorCode());
            else
                throw exc;
        }

    }

}
