package ru.yandex.chemodan.app.docviewer.convert.pdf;

import java.net.URL;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.AbstractSpringAwareTest;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.ConvertManager;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.misc.io.devnull.DevNullOutputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.test.Assert;

@RunWith(value = Parameterized.class)
public class PdfConverterSuiteTest extends AbstractSpringAwareTest {

    @Parameters
    public static Collection<Object[]> data() {
        Object[][] data = new Object[][] { { TargetType.PDF }, { TargetType.PLAIN_TEXT } };
        return Arrays.asList(data);
    }

    @Autowired
    private ConvertManager convertManager;

    private final TargetType convertTargetType;

    @Autowired
    private PdfConverter pdfConverter;

    public PdfConverterSuiteTest(TargetType convertTargetType) {
        this.convertTargetType = convertTargetType;
    }

    private void test(final URL doc, Option<String> password) {
        convertManager.convertCommon(new UrlInputStreamSource(doc), MimeTypes.MIME_PDF,
                pdfConverter, convertTargetType, new DevNullOutputStreamSource(), password);
    }

    private void test(final URL doc) {
        test(doc, Option.empty());
    }

    @Test
    public void test_Adobe_Acrobat_1_3_001p() {
        test(TestResources.Adobe_Acrobat_1_3_001p);
    }

    @Test
    public void test_Adobe_Acrobat_1_3_001p_2columns() {
        test(TestResources.Adobe_Acrobat_1_3_001p_2columns);
    }

    @Test
    public void test_Adobe_Acrobat_1_4_001p_limited() {
        test(TestResources.Adobe_Acrobat_1_4_001p_limited);
    }

    @Test
    public void test_Adobe_Acrobat_1_4_001p_password_exception() {
        try {
            test(TestResources.Adobe_Acrobat_1_4_001p_password);
            Assert.A.fail("UserException expected");
        } catch (UserException exc) {
            Assert.A.equals(ErrorCode.FILE_IS_PASSWORD_PROTECTED, exc.getErrorCode());
        }
    }

    @Test
    public void test_Adobe_Acrobat_1_4_001p_password_extract() {
        test(TestResources.Adobe_Acrobat_1_4_001p_password, Option.of("password"));
    }

    @Test
    public void test_Adobe_Acrobat_1_4_004p() {
        test(TestResources.Adobe_Acrobat_1_4_004p);
    }

    @Test
    public void test_Adobe_Acrobat_1_5_114p() {
        test(TestResources.Adobe_Acrobat_1_5_114p);
    }

}
