package ru.yandex.chemodan.app.docviewer.convert.pdf;

import java.awt.Image;
import java.io.ByteArrayInputStream;
import java.util.Arrays;

import javax.imageio.ImageIO;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.utils.ByteArrayOutputStreamSource;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.UriUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.PdfUtils;
import ru.yandex.chemodan.app.docviewer.utils.pdf.image.PdfRenderTargetTypeHolder;
import ru.yandex.chemodan.app.docviewer.web.DocviewerWebSpringTestBase;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.test.Assert;

public class PdfConverterTest extends DocviewerWebSpringTestBase {

    @Autowired
    private TestManager testManager;

    @Autowired
    private PdfConverter pdfConverter;

    @Autowired
    private PdfRenderTargetTypeHolder pdfRenderTargetTypeHolder;


    private static final String FILE_ID =
        "2713c-4wmgo6byuuiljaz8a47x76l3el3jswa982n4dc09tunqclydg0a4ly60xrjuunxp808h1l7mpiar967nufounpgqnzexljqqhc1";

    @Test
    public void test2columns() throws Exception {
        final ByteArrayOutputStreamSource baoss = new ByteArrayOutputStreamSource();

        pdfConverter.convert(new UrlInputStreamSource(TestResources.Adobe_Acrobat_1_3_001p_2columns),
                "", TargetType.PLAIN_TEXT, baoss, Option.empty());
        final String text = new String(baoss.getByteArray(), "utf-8");

        Assert.assertFalse(text.contains("HJJLZZ"));
    }

    @Test
    public void testEncoding() throws Exception {
        final ByteArrayOutputStreamSource baoss = new ByteArrayOutputStreamSource();

        pdfConverter.convert(new UrlInputStreamSource(TestResources.Adobe_Acrobat_1_4_009p_encoding),
                "", TargetType.PLAIN_TEXT, baoss, Option.empty());
        final String text = new String(baoss.getByteArray(), "utf-8");

        Assert.assertTrue(text.contains("Счёт"));
        Assert.assertTrue(text.contains("Жиделёв Николай Борисович"));
    }

    @Test
    public void testHtmlWithImages() throws Exception {
        final String fileId = testManager.makeAvailable(PassportUidOrZero.zero(), UriUtils.toUrlString(TestResources.Adobe_Acrobat_1_5_114p), TargetType.HTML_WITH_IMAGES);

        String ext = pdfRenderTargetTypeHolder.getTargetType().value();
        {
            byte[] htmlPageInfo = ApacheHttpClientUtils
                    .download("http://localhost:32405/htmlwithimagespageinfo?uid=0&id=" + fileId
                            + "&page=1");
            String response = new String(htmlPageInfo);

            Assert.assertContains(response, ".p1{position:relative;margin:auto;}");
            String expectedUrl = "./htmlimage?id=" + FILE_ID + "&amp;width=1024" + "&amp;name=bg-0." + ext;
            Assert.assertContains(response, "<img height=\"1448\" src=\"" + expectedUrl + "\" width=\"1024\"/>");
        }

        {
            byte[] imageData = ApacheHttpClientUtils
                    .download("http://localhost:32405/htmlimage?uid=0&id=" + FILE_ID
                            + "&width=1024" + "&name=bg-0." + ext);
            Image image = ImageIO.read(new ByteArrayInputStream(imageData));
            Assert.assertEquals(1024, image.getWidth(null));
            Assert.A.contained(image.getHeight(null),
                    Arrays.asList(1448, 1449));
        }

    }

    @Test
    public void testHtmlWithImages_havePicture() throws Exception {
        testManager.makeAvailable(PassportUidOrZero.zero(), UriUtils.toUrlString(TestResources.Adobe_Acrobat_1_5_114p), TargetType.HTML_WITH_IMAGES);

        String ext = pdfRenderTargetTypeHolder.getTargetType().value();
        byte[] imageData = ApacheHttpClientUtils
                .download("http://localhost:32405/htmlimage?uid=0&id=" + FILE_ID
                        + "&width=1024" + "&name=bg-14." + ext);
        Image image = ImageIO.read(new ByteArrayInputStream(imageData));
        Assert.assertEquals(1024, image.getWidth(null));
        Assert.A.contained(image.getHeight(null),
                Arrays.asList(1448, 1449));

        Assert.assertTrue("Image file too short: " + imageData.length, imageData.length > 2500);
    }

    @Test
    public void testHtmlWithImages_zoom() throws Exception {
        final String fileId = testManager.makeAvailable(PassportUidOrZero.zero(), UriUtils.toUrlString(TestResources.Adobe_Acrobat_1_5_114p), TargetType.HTML_WITH_IMAGES);

        String ext = pdfRenderTargetTypeHolder.getTargetType().value();
        {
            byte[] htmlPageInfo = ApacheHttpClientUtils
                    .download("http://localhost:32405/htmlwithimagespageinfo?uid=0&id=" + fileId
                            + "&page=1&width=2048");
            String response = new String(htmlPageInfo);

            Assert.assertContains(response, ".p1{position:relative;margin:auto;}");
            String expectedUrl = "./htmlimage?id=" + FILE_ID + "&amp;width=2048" + "&amp;name=bg-0." + ext;
            Assert.assertContains(response, "<img height=\"2896\" src=\"" + expectedUrl + "\" width=\"2048\"/>");
        }

        {
            byte[] imageData = ApacheHttpClientUtils
                    .download("http://localhost:32405/htmlimage?uid=0&id=" + FILE_ID
                            + "&width=2048" + "&name=bg-0." + ext);
            Image image = ImageIO.read(new ByteArrayInputStream(imageData));
            Assert.isTrue(checkIntBounds(2048, image.getWidth(null), 1));
            Assert.isTrue(checkIntBounds(2896, image.getHeight(null), 2));
        }

    }

    @Test
    public void testInvoice() throws Exception {
        final ByteArrayOutputStreamSource baoss = new ByteArrayOutputStreamSource();

        final UrlInputStreamSource pdfSource = new UrlInputStreamSource(
                TestResources.Adobe_Acrobat_1_5_001p_invoice);
        pdfConverter.convert(pdfSource, "", TargetType.PLAIN_TEXT, baoss, Option.empty());
        final String text = new String(baoss.getByteArray(), "utf-8");

        Assert.assertTrue(text.contains("Volksbank"));

        String textNoText = FileUtils.withEmptyTemporaryFile("notext", ".pdf",
                tempNotextPdfFile -> {
                    PdfUtils.filterText(pdfSource, 1, new FileOutputStreamSource(
                            tempNotextPdfFile));
                    return PdfUtils.withExistingDocument(tempNotextPdfFile, true,
                            PdfUtils::stripText);
                });
        Assert.assertFalse(textNoText.contains("Volksbank"));
    }

    @Test
    public void convertToPdfFirstPageEncryptedFile() {
        FileUtils.withEmptyTemporaryFile("tmp", ".pdf", result -> {
            pdfConverter.convert(new UrlInputStreamSource(TestResources.class.getResource("test/pdf/11700.pdf")),
                    "", TargetType.PREVIEW, result.asOutputStreamTool(), Option.empty());

            PdfUtils.withExistingDocument(result.asInputStreamTool(), true, document -> {
                Assert.equals(1, document.getNumberOfPages());
            });
        });
    }

    private boolean checkIntBounds(int value, int expected, int div) {
        return expected - div <= value && value <= expected + div;
    }
}
