package ru.yandex.chemodan.app.docviewer.convert.xls;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import javax.xml.parsers.DocumentBuilderFactory;

import eu.medsea.mimeutil.MimeType;
import eu.medsea.mimeutil.MimeUtil2;
import eu.medsea.mimeutil.detector.MagicMimeMimeDetector;
import org.apache.poi.hssf.OldExcelFormatException;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.docviewer.AbstractSpringAwareTest;
import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.MimeDetector;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.states.ErrorCode;
import ru.yandex.chemodan.app.docviewer.states.UserException;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.chemodan.app.docviewer.utils.ZipEntryInputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileInputStreamSource;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.test.Assert;

@RunWith(value = Parameterized.class)
public class PoiExamplesTest extends AbstractSpringAwareTest {

    private static final Logger logger = LoggerFactory.getLogger(PoiExamplesTest.class);

    private static final List<String> filesToProcess;

    private static File resultsDirectory = new File("target/test-results/poi-examples/xls");

    private static final Set<String> SKIP = Collections.unmodifiableSet(new HashSet<>(Arrays
            .asList( //
                    "",//
                    "43493.xls",// WONTFIX -- really messed up file
                    "password.xls", // encrypted
                    "" //
            )));

    static {
        filesToProcess = new ArrayList<>();

        final MimeUtil2 mimeUtil = new MimeUtil2();
        mimeUtil.registerMimeDetector(MagicMimeMimeDetector.class.getName());

        new File2(resultsDirectory).deleteRecursive();
        resultsDirectory.mkdirs();

        FileUtils.withZipFile(new UrlInputStreamSource(
                TestResources.Apache_Poi_Examples_Spreadsheets), new Function1V<ZipFile>() {
            @Override
            public void apply(ZipFile zipFile) {
                for (Enumeration<? extends ZipEntry> enumeration = zipFile.entries(); enumeration
                        .hasMoreElements();)
                {
                    ZipEntry zipEntry = enumeration.nextElement();
                    if (SKIP.contains(zipEntry.getName()))
                        continue;

                    File sourceXls = new File(resultsDirectory, zipEntry.getName());

                    final ZipEntryInputStreamSource zipEntryInputStreamSource = new ZipEntryInputStreamSource(
                            zipFile, zipEntry);

                    final Collection<MimeType> mimeTypes = mimeUtil
                            .getMimeTypes(zipEntryInputStreamSource.readBytes());
                    if (Cf.wrap(mimeTypes).map(ObjectUtils.toStringM())
                            .map(MimeDetector.normalizeF()).containsTs(MimeTypes.MIME_ARCHIVE_ZIP))
                        continue;

                    zipEntryInputStreamSource.readTo(new File2(sourceXls));

                    if (!XlsUtils.isSupported(new FileInputStreamSource(sourceXls), Option.empty())) {
                        logger.debug("File '{}' extracted but skipped because too complex "
                                + "(contains OLE or images)", zipEntry.getName());
                        continue;
                    }

                    logger.debug("File '{}' extracted", zipEntry.getName());
                    filesToProcess.add(zipEntry.getName());
                }
            }
        });

        Collections.sort(filesToProcess);
    }

    @Parameters
    public static Collection<Object[]> data() {
        return Cf.wrap(filesToProcess).map(a -> new Object[] { a });
    }

    private final DocumentBuilderFactory documentBuilderFactory;

    private final String entryName;

    @Autowired
    private MimeDetector mimeDetector;

    @Autowired
    private XlsConverter xlsConverter;

    public PoiExamplesTest(String entryName) {
        this.entryName = entryName;

        this.documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true);
    }

    @Test
    public void testHtmlOnly() {
        File inputFile = new File(resultsDirectory, entryName);

        try {
            File outputZipFile = new File(inputFile.getParentFile(), inputFile.getName() + ".zip");

            logger.info("Converting file '{}' to '{}'...",
                    new Object[] { inputFile, outputZipFile });

            xlsConverter.convert(new FileInputStreamSource(inputFile),
                    MimeTypes.MIME_MICROSOFT_EXCEL, TargetType.HTML_ONLY,
                    new FileOutputStreamSource(outputZipFile), Option.empty());

            logger.debug("File '{}' converted to '{}'", new Object[] { inputFile, outputZipFile });
        } catch (OldExcelFormatException exc) {
            Assert.A.equals("46904.xls", this.entryName);
        } catch (UserException exc) {
            Assert.A.equals(ErrorCode.FILE_IS_PASSWORD_PROTECTED, exc.getErrorCode());
        }
    }

    public void testMimeDetection() {
        File inputFile = new File(resultsDirectory, entryName);
        String detectedMime = mimeDetector.getMimeType(new FileInputStreamSource(inputFile));

        Assert.A.equals(MimeTypes.MIME_MICROSOFT_EXCEL, detectedMime);
    }

    // @Test
    // public void testPdf() {
    // XlsConverter xlsConverter = mockXlsConverter();
    // File inputFile = new File(resultsDirectory, entryName);
    //
    // try {
    // File outputPdfFile = new File(inputFile.getParentFile(),
    // inputFile.getName() + ".pdf");
    //
    // logger.info("Converting file '{}' to '{}'...",
    // new Object[] { inputFile, outputPdfFile });
    //
    // xlsConverter.convert(new FutureTask<Object>(new Callable<Object>() {
    // @Override
    // public Object call() throws Exception {
    // return null;
    // }
    // }), new FileInputStreamSource(inputFile), MimeTypes.MIME_MICROSOFT_EXCEL,
    // TargetType.HTML_ONLY, new FileOutputStreamSource(outputPdfFile),
    // new HashMap<String, Object>());
    //
    // logger.debug("File '{}' converted to '{}'", new Object[] { inputFile,
    // outputPdfFile });
    //
    // } catch (Exception exc) {
    // throw ExceptionUtils.translate(exc);
    // }
    // }
}
