package ru.yandex.chemodan.app.docviewer.states;

import java.util.function.Supplier;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Ignore;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.docviewer.DocviewerSpringTestBase;
import ru.yandex.chemodan.app.docviewer.TestManager;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.convert.TargetType;
import ru.yandex.chemodan.app.docviewer.copy.ActualUri;
import ru.yandex.chemodan.app.docviewer.copy.CopiedFileInfo;
import ru.yandex.chemodan.app.docviewer.copy.DocumentSourceInfo;
import ru.yandex.chemodan.app.docviewer.copy.UriHelper;
import ru.yandex.chemodan.app.docviewer.utils.FileCopy;
import ru.yandex.chemodan.app.docviewer.utils.FileUtils;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author vlsergey
 * @author akirakozov
 */
public class StateMachineImplTest extends DocviewerSpringTestBase {

    @Autowired
    private StateMachineImpl stateMachine;

    @Autowired
    private TestManager testManager;

    @Autowired
    private UriHelper uriHelper;

    @Test
    public void detectContentType() throws Exception {
        Assert.equals("text/plain", doDetectContentType("file:///123.txt", "application/octet-stream", "123.txt"));
        Assert.equals("text/plain", doDetectContentType("file:///123_no_txt_ext", "application/octet-stream", "123.txt"));
        Assert.equals("text/plain", doDetectContentType("file:///some.txt", "text/plain", "some.txt", "#!/bin/sh\nrm -rf *\n"));
    }

    @Test
    public void testPrecedence() throws Exception {
        Assert.equals("text/csv", doDetectContentType("file:///1.csv", "text/csv", "1.csv"));
        Assert.equals("text/csv", doDetectContentType("file:///csvContents.xls", "text/csv", "csvContents.xls"));
    }

    @Test
    @Ignore("flaky test")
    public void processWithoutConvertAndThanConvert() {
        ActualUri url = new ActualUri(TestResources.Adobe_Acrobat_1_3_001p);
        TargetType target = TargetType.HTML_WITH_IMAGES;

        testManager.cleanupUri(PassportUidOrZero.zero(), url.getUriString());
        Assert.equals(State.NOT_STARTED, stateMachine.getState(url, target));

        // process without converting
        DocumentSourceInfo source = DocumentSourceInfo.builder().originalUrl(url.getUriString()).build();
        stateMachine.onStart(source, "test", Option.empty(),
                target, "", Cf.list("application/pdf"), Instant.now(), false);
        ThreadUtils.sleep(Duration.millis(300));

        Assert.equals(State.COPIED, stateMachine.getState(url, target));

        // process with converting after copied state
        stateMachine.onStart(source, "test", Option.empty(), target, "", Cf.list(), Instant.now(), false);
        testManager.makeAvailable(PassportUidOrZero.zero(), url.getUriString(), target);
    }

    private String doDetectContentType(
            String uri, String reportedContentType, String contentDispositionFilename)
            throws Exception
    {
        return doDetectContentType(uri, reportedContentType, contentDispositionFilename, "123");
    }

    private String doDetectContentType(
            String uri, String reportedContentType, String contentDispositionFilename,
            String fileContent) throws Exception
    {
        final File2 temporaryFile = FileUtils.createTempFile("file", ".whatever",
                new ByteArrayInputStreamSource(fileContent.getBytes("UTF-8")));

        return stateMachine.detectContentType(new CopiedFileInfo(
                new ActualUri(uri),
                Option.of(reportedContentType),
                Option.of(contentDispositionFilename),
                new FileCopy(temporaryFile),
                "someId",
                true,
                Option.empty(),
                null, ""));
    }

    @Test
    public void testStartMailAttachment() throws InterruptedException {
        String url = "ya-mail://1000005.5181427.426298960252555950252128867516/1.2";
        PassportUidOrZero uid = new PassportUidOrZero(42);
        TargetType type = TargetType.HTML_WITH_IMAGES;
        DocumentSourceInfo source = DocumentSourceInfo.builder().originalUrl(url).uid(uid).build();
        ActualUri actualUri = uriHelper.rewrite(source);
        testManager.cleanupUri(uid, actualUri.getUriString());
        stateMachine.onStart(source, "test", Option.empty(), type, "", Cf.list(), Instant.now(), false);
        Supplier<State> state = () -> stateMachine.getState(actualUri, type);
        for (int i = 0; state.get() != State.COPIED && i < 30; ++i, Thread.sleep(1000));
        Assert.equals(State.COPIED, state.get());
    }

}
