package ru.yandex.chemodan.app.docviewer.utils.pdf.image;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.URL;

import javax.imageio.ImageIO;

import org.joda.time.Duration;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.chemodan.app.docviewer.MimeTypes;
import ru.yandex.chemodan.app.docviewer.TestResources;
import ru.yandex.chemodan.app.docviewer.adapters.poppler.ResizeOption;
import ru.yandex.chemodan.app.docviewer.utils.DimensionO;
import ru.yandex.chemodan.app.docviewer.web.DocviewerWebSpringTestBase;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.file.FileOutputStreamSource;
import ru.yandex.misc.io.url.UrlInputStreamSource;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

public class PdfRendererTest extends DocviewerWebSpringTestBase {

    private static final Logger logger = LoggerFactory.getLogger(PdfRendererTest.class);

    @Autowired
    @Qualifier("pdfRenderer")
    private PdfRenderer pdfRenderer;

    private File2 tmpDir;
    private File2 tmpFile;

    @Before
    public void init() {
        tmpDir = File2.createNewTmpDir();
        tmpFile = new File2(tmpDir, "tmp");
    }

    @After
    public void destroy() {
        tmpDir.deleteRecursive();
    }

    @Test
    public void testToImageLandscapeBig() {
        test(TestResources.Adobe_Acrobat_1_4_004p, 2, 2000, 1414);
    }

    @Test
    public void testToImageLandscapeSmall() {
        test(TestResources.Adobe_Acrobat_1_4_004p, 2, 100, 71);
    }

    @Test
    public void testToImagePortraitBig() {
        test(TestResources.Adobe_Acrobat_1_5_114p, 34, 2000, 2829);
    }

    @Test
    public void testToImagePortraitSmall() {
        test(TestResources.Adobe_Acrobat_1_5_114p, 34, 100, 141);
    }

    @Test
    public void checkTimeout() {
        try {
            ThreadLocalTimeout.push(Duration.millis(40));
            test(TestResources.Adobe_Acrobat_1_4_004p, 2, 900, 636);
            Assert.fail("Timeout expected");
        } catch (ThreadLocalTimeoutException exc) {
            // Ok
        }
    }

    @Test
    public void shouldAcceptTargetType() {
        UrlInputStreamSource source = new UrlInputStreamSource(TestResources.Adobe_Acrobat_1_4_001p);

        RenderedImageInfo info = pdfRenderer.render(source, 1,
                ResizeOption.scale(DimensionO.cons(900)), PdfRenderTargetType.JPG, new FileOutputStreamSource(tmpFile));

        Assert.equals(MimeTypes.MIME_IMAGE_JPEG, info.contentType);
    }

    private void test(final URL url, final int page, final int width, final int expectedHeight) {
        final UrlInputStreamSource pdfDocumentSource = new UrlInputStreamSource(url);
        try {
            RenderedImageInfo renderedImageInfo = pdfRenderer.render(pdfDocumentSource,
                    page, ResizeOption.scale(DimensionO.cons(width)), new FileOutputStreamSource(tmpFile));

            logger.info("Temporary result image stored in " + tmpFile.getAbsolutePath());

            // poppler renderer allows square bounding box and can give a mistake even for width
            Assert.le(Math.abs(width - renderedImageInfo.width), 1);
            Assert.le(Math.abs(expectedHeight - renderedImageInfo.height), 1);

            BufferedImage bufferedImage = ImageIO.read(tmpFile.getFile());

            Assert.equals(renderedImageInfo.width, bufferedImage.getWidth());
            Assert.equals(renderedImageInfo.height, bufferedImage.getHeight());
        } catch (IOException exc) {
            throw IoUtils.translate(exc);
        }
    }

}
