package ru.yandex.chemodan.app.stat;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.PreDestroy;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.stat.limits.LimitsManager;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelAndAuth;
import ru.yandex.chemodan.app.stat.storage.DownloadStat;
import ru.yandex.chemodan.app.stat.storage.DownloadStatDao;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.OneChannelStats;
import ru.yandex.chemodan.app.stat.storage.SpooledCountryMediatypeStatsWriter;
import ru.yandex.chemodan.app.stat.storage.SpooledPeriodStatsWriter;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.mime.detect.MediaType;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Meter;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.MeterMapWithDistribution;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.time.InstantInterval;

import static ru.yandex.bolts.collection.Option.of;
import static ru.yandex.chemodan.app.stat.DownloadStatChannels.FOLDER;
import static ru.yandex.chemodan.app.stat.DownloadStatChannels.SIMPLE;

/**
 * @author Lev Tolmachev
 */
public class DiskStatManager implements MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(DiskStatManager.class);

    private final ExecutorService checkLimitsExecutor;

    private final DownloadStatDao downloadStatDao;
    private final LimitsManager limitsManager;
    private final SpooledPeriodStatsWriter periodStats;
    private final SpooledCountryMediatypeStatsWriter mediatypeStats;

    @MonicaMetric
    @GroupByDefault
    private final MeterMap views = new MeterMapWithDistribution();
    @MonicaMetric
    @GroupByDefault
    private final MeterMap traffic = new MeterMapWithDistribution();

    @MonicaMetric
    @GroupByDefault
    private final MeterMap viewsPublic = new MeterMapWithDistribution();
    @MonicaMetric
    @GroupByDefault
    private final MeterMap trafficPublic = new MeterMapWithDistribution();

    @MonicaMetric
    @GroupByDefault
    private final MeterMap viewsAuth = new MeterMapWithDistribution();
    @MonicaMetric
    @GroupByDefault
    private final MeterMap trafficAuth = new MeterMapWithDistribution();

    @MonicaMetric
    @GroupByDefault
    private final Meter viewsSum = new Meter();
    @MonicaMetric
    @GroupByDefault
    private final Meter trafficSum = new Meter();

    @MonicaMetric
    @GroupByDefault
    private final Meter viewsPublicSum = new Meter();
    @MonicaMetric
    @GroupByDefault
    private final Meter trafficPublicSum = new Meter();

    @MonicaMetric
    @GroupByDefault
    private final Meter viewsAuthSum = new Meter();
    @MonicaMetric
    @GroupByDefault
    private final Meter trafficAuthSum = new Meter();

    public DiskStatManager(DownloadStatDao downloadStatDao, LimitsManager limitsManager,
            SpooledPeriodStatsWriter periodStats, SpooledCountryMediatypeStatsWriter mediatypeStats,
            int poolSize, int queueCapacity)
    {
        this(downloadStatDao, limitsManager, periodStats, mediatypeStats,
                new ThreadPoolExecutor(poolSize, poolSize, 0L, TimeUnit.MILLISECONDS,
                        new LinkedBlockingQueue<>(queueCapacity), DiskStatManager::putToQueueOrThrow));
    }

    DiskStatManager(DownloadStatDao downloadStatDao, LimitsManager limitsManager,
            SpooledPeriodStatsWriter periodStats, SpooledCountryMediatypeStatsWriter mediatypeStats,
            ExecutorService checkLimitsExecutor)
    {
        this.downloadStatDao = downloadStatDao;
        this.limitsManager = limitsManager;
        this.periodStats = periodStats;
        this.mediatypeStats = mediatypeStats;
        this.checkLimitsExecutor = checkLimitsExecutor;
    }

    @PreDestroy
    public void destroy() {
        checkLimitsExecutor.shutdown();
    }

    public MapF<String, OneChannelStats> getFileStat(final DownloadStatId id, InstantInterval period) {
        return downloadStatDao.getDownloadStats(id, period.getStart())
                .map(DownloadStat.getChannelStatsF())
                .getOrElse(Cf::hashMap);
    }

    public DownloadStat incHashDownloadStat(String hash, OneChannelStats statsDelta) {
        return incDownloadStat(new DownloadStatId(hash), Option.empty(), Option.empty(),
                FOLDER, statsDelta, Option.empty());
    }

    public DownloadStat incFileDownloadStat(
            MpfsHid hid, Option<String> hash, Option<String> name, Option<String> mediatype, String channel,
            OneChannelStats statsDelta)
    {
        return incDownloadStat(
                new DownloadStatId(hid), name, mediatype, channel, statsDelta,
                of((stats, restrictions) -> {
                    limitsManager.checkAntiPornoLimits(stats, hash, name);
                    if (restrictions.isNotEmpty() && hash.isPresent()) {
                        //if file access limited and exists hash(=folder) - limit access to hash(=folder)
                        ListF<ChannelAndAuth> hashRestrictions =
                                restrictions.map(r -> new ChannelAndAuth(FOLDER, r.auth)).stableUnique();
                        logger.info("Limiting access to folder with hash '{}': {}", hash.get(), hashRestrictions);
                        limitsManager.limitAccess(new DownloadStatId(hash.get()), hashRestrictions);
                    }
                })
        );
    }

    private DownloadStat incDownloadStat(DownloadStatId statId, Option<String> name,
            Option<String> mediatype, String channel, OneChannelStats statsDelta,
            Option<Function2V<DownloadStat, ListF<ChannelAndAuth>>> checkLimitsCallback)
    {
        logger.info("Increasing stats. id={}, channel={}, name={}, mediatype={}, stats={}",
                statId, channel, name, mediatype, statsDelta);

        if (SIMPLE.equals(channel)) {
            ListF<MetricName> names = mediatype.map(s -> new MetricName(getMediatype(of(s))));

            trafficSum.inc(statsDelta.getTraffic().toBytes());
            viewsSum.inc();

            views.inc(statsDelta.getViews(), names);
            traffic.inc(statsDelta.getTraffic().toBytes(), names);

            mediatypeStats.inc(getMediatype(mediatype), statsDelta);

            viewsAuth.inc(statsDelta.getAuthStats().getViews(), names);
            trafficAuth.inc(statsDelta.getAuthStats().getTraffic().toBytes(), names);
            trafficAuthSum.inc(statsDelta.getAuthStats().getTraffic().toBytes());
            viewsAuthSum.inc(statsDelta.getAuthStats().getViews());

            viewsPublic.inc(statsDelta.getPublicStats().getViews(), names);
            trafficPublic.inc(statsDelta.getPublicStats().getTraffic().toBytes(), names);
            trafficPublicSum.inc(statsDelta.getPublicStats().getTraffic().toBytes());
            viewsPublicSum.inc(statsDelta.getPublicStats().getViews());
        }

        DownloadStat stats = downloadStatDao.incStats(statId, channel, statsDelta);

        checkLimitsExecutor.submit(() -> {
            try {
                ListF<ChannelAndAuth> restrictions = limitsManager.checkLimits(stats, mediatype);
                checkLimitsCallback.ifPresent(f -> f.accept(stats, restrictions));
            } catch (RuntimeException e) {
                logger.error("Failed to check file limits: {}", e);
            }
        });
        periodStats.inc(channel, statsDelta);
        return stats;
    }

    private static String getMediatype(Option<String> mediatype) {
        // hack for video_short and video_long from videodisk
        if (mediatype.isPresent() && mediatype.get().startsWith("video_")) {
            mediatype = of("video");
        }
        return MediaType.R.valueOfO(mediatype.getOrElse(MediaType.UNKNOWN.getName()))
                .getOrElse(MediaType.UNKNOWN).getName();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "stat-manager",
                new MetricName("stat-manager", "store"),
                "Statistic manager"
        );
    }

    private static void putToQueueOrThrow(Runnable r, ThreadPoolExecutor executor) {
        try {
            executor.getQueue().put(r);
        } catch (InterruptedException e) {
            throw ExceptionUtils.translate(e);
        }
    }
}
