package ru.yandex.chemodan.app.stat;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.limits.LimitsManager;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelAndAuth;
import ru.yandex.chemodan.app.stat.storage.DownloadStat;
import ru.yandex.chemodan.app.stat.storage.DownloadStatDao;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.OneChannelStats;
import ru.yandex.chemodan.app.stat.storage.SpooledCountryMediatypeStatsWriter;
import ru.yandex.chemodan.app.stat.storage.SpooledPeriodStatsWriter;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.thread.executor.SyncExecutor;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;
import static ru.yandex.chemodan.app.stat.DownloadStatChannels.FOLDER;
import static ru.yandex.chemodan.app.stat.DownloadStatChannels.SIMPLE;
import static ru.yandex.chemodan.app.stat.storage.OneChannelStats.of;
import static ru.yandex.misc.test.Assert.assertContains;
import static ru.yandex.misc.test.Assert.assertEquals;
import static ru.yandex.misc.test.Assert.assertHasSize;

public class DiskStatManagerTest {
    private DiskStatManager diskStatManager;

    @Mock
    private DownloadStatDao downloadStatDao;
    @Mock
    private LimitsManager limitsManager;
    @Mock
    private SpooledPeriodStatsWriter periodStats;
    @Mock
    private SpooledCountryMediatypeStatsWriter countryMediatypeStatsWriter;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);

        when(downloadStatDao.incStats(any(), any(), any())).thenAnswer(
                invocation -> new DownloadStat(
                        invocation.getArgument(0),
                        Cf.map(invocation.getArgument(1), invocation.getArgument(2))));

        diskStatManager = new DiskStatManager(
                downloadStatDao, limitsManager, periodStats, countryMediatypeStatsWriter, new SyncExecutor());
    }

    @After
    public void destroy() {
        diskStatManager.destroy();
    }


    @Test
    public void testIncFileDownloadStat() {
        when(limitsManager.checkLimits(any(), any())).thenReturn(Cf.list());
        MpfsHid hid = new MpfsHid("someHid");
        String hash = "someHash";

        DownloadStat stats = callIncFileDownloadStat(hid, hash);

        Mockito.verify(limitsManager).checkLimits(stats, Option.empty());
        Mockito.verify(limitsManager).checkAntiPornoLimits(stats, Option.of(hash), Option.empty());
        Mockito.verifyNoMoreInteractions(limitsManager);
    }

    @Test
    public void testIncFileDownloadStatWithFolderLocking() {
        when(limitsManager.checkLimits(any(), any())).thenReturn(Cf.list(new ChannelAndAuth(SIMPLE, false)));
        MpfsHid hid = new MpfsHid("someHid");
        String hash = "someHash";

        DownloadStat stats = callIncFileDownloadStat(hid, hash);

        Mockito.verify(limitsManager).checkLimits(stats, Option.empty());
        Mockito.verify(limitsManager).checkAntiPornoLimits(stats, Option.of(hash), Option.empty());
        Mockito.verify(limitsManager).limitAccess(new DownloadStatId(hash), Cf.list(new ChannelAndAuth(FOLDER, false)));
        Mockito.verifyNoMoreInteractions(limitsManager);
    }

    @Test
    public void testIncHashDownloadStat() {
        when(limitsManager.checkLimits(any(), any())).thenReturn(Cf.list());
        String hash = "someHash";
        DownloadStatId expectedStatId = new DownloadStatId(hash);


        OneChannelStats deltaStats = of(false, DataSize.MEGABYTE, 10);
        DownloadStat stats = diskStatManager.incHashDownloadStat(hash, deltaStats);
        verifyResponse(stats, expectedStatId, FOLDER, deltaStats);

        Mockito.verify(downloadStatDao).incStats(expectedStatId, FOLDER, deltaStats);
        Mockito.verifyNoMoreInteractions(downloadStatDao);

        Mockito.verify(limitsManager).checkLimits(stats, Option.empty());
        Mockito.verifyNoMoreInteractions(limitsManager);
    }

    private DownloadStat callIncFileDownloadStat(MpfsHid hid, String hash) {
        DownloadStatId expectedStatId = new DownloadStatId(hid);
        OneChannelStats deltaStats = of(false, DataSize.MEGABYTE, 10);

        DownloadStat stats = diskStatManager.incFileDownloadStat(hid, Option.of(hash), Option.empty(),
                Option.empty(), SIMPLE, deltaStats);
        verifyResponse(stats, expectedStatId, SIMPLE, deltaStats);

        Mockito.verify(downloadStatDao).incStats(expectedStatId, SIMPLE, deltaStats);
        Mockito.verifyNoMoreInteractions(downloadStatDao);

        return stats;
    }

    private void verifyResponse(DownloadStat actual, DownloadStatId expectedId,
            String expectedChannel, OneChannelStats expectedChannelStats)
    {
        assertEquals(expectedId, actual.getId());
        assertHasSize(1, actual.getChannelStats().values());
        assertContains(actual.getChannelStats().keys(), expectedChannel);

        OneChannelStats channelStats = actual.getChannelStats().getTs(expectedChannel);
        assertEquals(expectedChannelStats.getPublicStats().getViews(), channelStats.getPublicStats().getViews());
        assertEquals(expectedChannelStats.getPublicStats().getTraffic(), channelStats.getPublicStats().getTraffic());
        assertEquals(expectedChannelStats.getAuthStats().getViews(), channelStats.getAuthStats().getViews());
        assertEquals(expectedChannelStats.getAuthStats().getTraffic(), channelStats.getAuthStats().getTraffic());
    }


}
