package ru.yandex.chemodan.app.stat.admin;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.DiskStatManager;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.chemodan.app.stat.limits.LimitedInfo;
import ru.yandex.chemodan.app.stat.limits.LimitsManager;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.DownloadStatType;
import ru.yandex.chemodan.app.stat.storage.OneChannelStats;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Lev Tolmachev
 */
@ActionContainer
public class FileStatsAdminPage {
    private static final Logger logger = LoggerFactory.getLogger(FileStatsAdminPage.class);

    private final Duration period;
    private final DiskStatManager diskStatManager;
    private final LimitsManager limitsManager;
    private final MpfsClient mpfsClient;

    public FileStatsAdminPage(Duration period, DiskStatManager diskStatManager, LimitsManager limitsManager,
            MpfsClient mpfsClient)
    {
        this.period = period;
        this.diskStatManager = diskStatManager;
        this.limitsManager = limitsManager;
        this.mpfsClient = mpfsClient;
    }

    @ZAction(defaultAction = true, file = "FileStatAdminPage.xsl")
    @Path("/file-stat")
    public Content content(
            @RequestParam(required = false, value = "id") String id,
            @RequestParam(required = false, value = "type") DownloadStatType type,
            @RequestParam(value = "periods") Option<Integer> periods)
    {
        if (StringUtils.isEmpty(id)) {
            return Content.EMPTY;
        }
        DownloadStatId statId = new DownloadStatId(id, type);
        MapF<String, MapF<String, OneChannelStats>> stats = Cf.hashMap();
        MapF<String, LimitedInfo> limits = Cf.hashMap();
        InstantInterval interval = TimeUtils.getCurrentPeriod(period);
        for (int i = 0; i < periods.getOrElse(7); i++) {
            stats.put(TimeUtils.intervalToString(interval), diskStatManager.getFileStat(statId, interval));
            limits.put(TimeUtils.intervalToString(interval), limitsManager.getLimitedInfo(statId, interval));
            interval = TimeUtils.getPreviousPeriod(interval);
        }
        return new Content(statId, stats, limits);
    }

    @ZAction(file = "FileStatAdminPage-info.xsl")
    @Path("/file-info")
    public MpfsFileInfo fileInfo(@RequestParam(value = "hid") String hid)
    {
        return mpfsClient.getFileInfoByHid(new MpfsHid(hid));
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    @XmlRootElement(name = "content")
    private static final class Content {
        private static final Content EMPTY = new Content("", DownloadStatType.HID, Cf.map(), Cf.map());

        private final String id;
        private final DownloadStatType type;
        private final MapF<String, MapF<String, OneChannelStats>> stats;
        private final MapF<String, LimitedInfo> limits;

        public Content(String id, DownloadStatType type,
                MapF<String, MapF<String, OneChannelStats>> stats,
                MapF<String, LimitedInfo> limits)
        {
            this.id = id;
            this.type = type;
            this.stats = stats;
            this.limits = limits;
        }

        private Content(DownloadStatId id, MapF<String, MapF<String, OneChannelStats>> stats,
                MapF<String, LimitedInfo> limits)
        {
            this(id.getIdString(), id.getType(), stats, limits);
        }
    }
}
