package ru.yandex.chemodan.app.stat.admin;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.chemodan.app.stat.limits.LimitedInfo;
import ru.yandex.chemodan.app.stat.limits.LimitsManager;
import ru.yandex.chemodan.app.stat.limits.block.BlockDownloadsManager;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.DownloadStatType;
import ru.yandex.chemodan.app.stat.web.ConvertToInterval;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Lev Tolmachev
 */
@ActionContainer
public class LimitedFilesAdminPage {

    private final Duration period;
    private final LimitsManager limitsManager;
    private final BlockDownloadsManager blockDownloadsManager;

    public LimitedFilesAdminPage(Duration period, LimitsManager limitsManager,
            BlockDownloadsManager blockDownloadsManager)
    {
        this.period = period;
        this.limitsManager = limitsManager;
        this.blockDownloadsManager = blockDownloadsManager;
    }

    @ZAction(file = "LimitedFilesAdminPage.xsl", defaultAction = true)
    @Path("/limited-files")
    public IndexContent index(@RequestParam(value = "periods") Option<Integer> periods)
    {
        InstantInterval interval = TimeUtils.getCurrentPeriod(period);
        MapF<String, Long> map = Cf.hashMap();
        int periodsActual = periods.getOrElse(7);
        for (int i = 0; i < periodsActual; i++) {
            map.put(TimeUtils.intervalToString(interval), limitsManager.getLimitedFilesCount(interval));
            interval = TimeUtils.getPreviousPeriod(interval);
        }
        return new IndexContent(map, periodsActual, blockDownloadsManager.getBlockedFilesCount());
    }

    @ZAction(file = "LimitedFilesAdminPage-period.xsl")
    @Path("/limited-files/period")
    public PeriodContent limitedInPeriod(
            @RequestParam(value = "period", customConverter = ConvertToInterval.class)
                    InstantInterval period)
    {
        return new PeriodContent(limitsManager.getLimitedFiles(period), blockDownloadsManager.getBlockedFilesCount());
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    @XmlRootElement(name = "content")
    private static final class IndexContent {
        public final MapF<String, Long> limited;
        public final long blockedFilesCount;
        public final int periods;

        private IndexContent(MapF<String, Long> limited, int periods, long blockedFilesCount) {
            this.periods = periods;
            this.limited = limited;
            this.blockedFilesCount = blockedFilesCount;
        }
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    @XmlRootElement(name = "content")
    private static final class PeriodContent {
        public final ListF<LimitedPojo> limited;
        public final long blockedFilesCount;

        private PeriodContent(MapF<DownloadStatId, LimitedInfo> limited, long blockedFilesCount) {
            this.blockedFilesCount = blockedFilesCount;
            this.limited =
                    limited.mapEntries(LimitedPojo::new);
        }
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    private static final class LimitedPojo {
        public final String key;
        public final DownloadStatType type;
        public final LimitedInfo value;
        public final Instant lastTime;

        private LimitedPojo(DownloadStatId downloadStatId, LimitedInfo value) {
            this.key = downloadStatId.getIdString();
            this.type = downloadStatId.getType();
            this.value = value;
            CollectionF<Instant> collection =
                    value.authLimited.values().plus(value.publicLimited.values());
            if (collection.isEmpty()) {
                lastTime = new Instant(0);
            } else {
                lastTime = collection.max();
            }
        }
    }
}
