package ru.yandex.chemodan.app.stat.antiporno;

import java.util.List;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.antiporno.processors.AbstractDetector;
import ru.yandex.chemodan.app.stat.antiporno.processors.AbstractFilter;
import ru.yandex.chemodan.app.stat.antiporno.processors.AbstractProcessor;
import ru.yandex.chemodan.app.stat.antiporno.scores.AutoAntiPornoScores;
import ru.yandex.chemodan.app.stat.moderate.ModerationLink;
import ru.yandex.chemodan.app.stat.moderate.ModerationQueueClient;
import ru.yandex.chemodan.app.stat.moderate.ModeratorItem;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;

import static ru.yandex.chemodan.app.stat.moderate.ModerationLink.LinkType.FILE;

/**
 * @author metal
 */
public class PornoChecker {
    private static final Logger logger = LoggerFactory.getLogger(PornoChecker.class);

    private final ListF<AbstractProcessor> processors;

    private final ModerationQueueClient moderationQueueClient;
    private final AntipornoCheckedFilesDao antipornoCheckedFilesDao;
    private final double antiPornoScoreLimit;

    public PornoChecker(List<AbstractFilter> filters, List<AbstractDetector> detectors,
            ModerationQueueClient moderationQueueClient, AntipornoCheckedFilesDao antipornoCheckedFilesDao,
            double antiPornoScoreLimit)
    {
        this.processors = Cf.arrayList();
        processors.addAll(filters);
        processors.addAll(detectors);
        this.moderationQueueClient = moderationQueueClient;
        this.antipornoCheckedFilesDao = antipornoCheckedFilesDao;
        this.antiPornoScoreLimit = antiPornoScoreLimit;
    }

    public void checkFileForPorno(MpfsHid hid, Option<String> hash, Option<String> name) {
        Ndc.Handle handle = null;
        try {
            handle = Ndc.push((StringUtils.isBlank(Ndc.get()) ? "" : "#") + "hid=" + hid);

            logger.info("Starting to check file for porno");

            if (!antipornoCheckedFilesDao.contains(hid)) {
                PornoCheckRecord record = new PornoCheckRecord(hid, hash, name);

                boolean inModerationQueue = checkAntipornoScoresAndAddToModerationQueue(hid, record);

                if (inModerationQueue || record.addToCheckedFilesCollection) {
                    logger.info("Inserting file to checked files collection");
                    antipornoCheckedFilesDao.insert(hid);
                }
            } else {
                logger.info("File was skipped, it was already checked for porno");
            }
        } catch (Exception e) {
            logger.error("Something goes wrong", e);
        } finally {
            if (handle != null) {
                handle.pop();
            }
        }
    }

    private boolean checkAntipornoScoresAndAddToModerationQueue(MpfsHid hid, PornoCheckRecord record) {
        boolean inModerationQueue = moderationQueueClient.containsAntipornoDetectedFile(hid);

        if (!inModerationQueue) {
            AutoAntiPornoScores scores = calculateAutoAntiPornoScores(record);

            if (scores.total >= antiPornoScoreLimit) {
                ListF<ModerationLink> links = Cf.arrayList();
                if (record.data.containsKeyTs("short_url")) {
                    String shortUrl = record.<String>getTypedData("short_url").get();
                    ModerationLink fileLink = new ModerationLink(FILE, shortUrl);
                    links.add(fileLink);
                }

                logger.info("Adding file to moderation queue with scores: {}", scores.toString());

                if (!moderationQueueClient.containsAntipornoDetectedFile(hid)) {
                    moderationQueueClient.addToQueue(new ModeratorItem(Instant.now(), scores.toString(),
                            links, ModeratorItem.Source.ANTI_PORNO, Option.of(hid)));
                }
                inModerationQueue = true;
            }

            logger.info("Checking for porno finished");
        } else {
            logger.info("File was skipped, it is already in moderation queue");
        }

        return inModerationQueue;
    }

    private AutoAntiPornoScores calculateAutoAntiPornoScores(PornoCheckRecord record) {
        for (AbstractProcessor processor : processors) {
            if (!processor.process(record)) {
                return new AutoAntiPornoScores(Cf.list());
            }
        }

        return new AutoAntiPornoScores(record.scores);
    }
}
