package ru.yandex.chemodan.app.stat.antiporno;

import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Matchers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.moderate.ModerationQueueClient;
import ru.yandex.chemodan.app.stat.moderate.ModeratorItem;
import ru.yandex.chemodan.mpfs.MpfsClientContextConfiguration;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static ru.yandex.chemodan.app.stat.antiporno.AntipornoTestsConfig.FAKE_HID_1;
import static ru.yandex.chemodan.app.stat.antiporno.AntipornoTestsConfig.FAKE_HID_2;

/**
 * @author metal
 */
@ContextConfiguration(classes = {
        MpfsClientContextConfiguration.class,
        AntipornoTestsConfig.class
})
public class PornoCheckerTest extends AbstractTest {


    @Autowired
    private PornoChecker pornoChecker;

    @Autowired
    private ModerationQueueClient mockedModerationQueueClient;

    @Autowired
    private AntipornoCheckedFilesDao mockedAntipornoCheckedFilesDao;

    @Test
    public void testDetectingByName() {
        pornoChecker.checkFileForPorno(FAKE_HID_1, Option.of("123"), Option.of("8yo lick.mp4"));

        ArgumentCaptor<ModeratorItem> moderatorItemArgumentCaptor = ArgumentCaptor.forClass(ModeratorItem.class);
        verify(mockedModerationQueueClient).addToQueue(moderatorItemArgumentCaptor.capture());
        ModeratorItem capturedItem = moderatorItemArgumentCaptor.getValue();

        Assert.equals(FAKE_HID_1, capturedItem.hid.get());
        Assert.equals(ModeratorItem.Source.ANTI_PORNO, capturedItem.source);
        Assert.equals("{\"total\":1.0,\"scores\":[{\"detector\":\"name\",\"value\":1.0,\"weight\":1.0,\"cause\":\"Name '8yo lick.mp4' contains regexp: .*[0-9]+yo\"}]}",
                capturedItem.description);

        pornoChecker.checkFileForPorno(FAKE_HID_1, Option.of("123"), Option.of("8yo lick.mp4"));

        verify(mockedModerationQueueClient, times(3)).containsAntipornoDetectedFile(Matchers.eq(FAKE_HID_1));
        verify(mockedModerationQueueClient, times(1)).addToQueue(any());
    }

    @Test
    public void testSkippingAlreadyCheckedFiles() {
        pornoChecker.checkFileForPorno(FAKE_HID_2, Option.of("123"), Option.of("video.mp4"));

        ArgumentCaptor<MpfsHid> hidArgumentCaptor = ArgumentCaptor.forClass(MpfsHid.class);
        verify(mockedAntipornoCheckedFilesDao).insert(hidArgumentCaptor.capture());
        MpfsHid capturedItem = hidArgumentCaptor.getValue();

        Assert.equals(FAKE_HID_2, capturedItem);

        pornoChecker.checkFileForPorno(FAKE_HID_2, Option.of("123"), Option.of("video.mp4"));

        verify(mockedAntipornoCheckedFilesDao, times(2)).contains(Matchers.eq(FAKE_HID_2));
        verify(mockedAntipornoCheckedFilesDao, times(1)).insert(Matchers.eq(FAKE_HID_2));
    }
}
