package ru.yandex.chemodan.app.stat.antiporno.collectors;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.antiporno.PornoCheckRecord;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMeta;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class MpfsResourceInfoCollector implements DataCollector {
    private static final Logger logger = LoggerFactory.getLogger(MpfsResourceInfoCollector.class);

    @Autowired
    private MpfsClient mpfsClient;

    @Override
    public void collect(PornoCheckRecord record) {
        if (record.data.containsKeyTs("hash") && !record.data.containsKeyTs("hasResourceInfo")) {
            String hash = record.<String>getTypedData("hash").get();
            try {
                Option<MpfsFileInfo> info = mpfsClient.getFileInfoOByPrivateHash(hash);
                if (info.isPresent()) {
                    record.data.put("hasResourceInfo", true);

                    MpfsFileInfo fileInfo = info.get();
                    record.data.put("type", fileInfo.type.get());

                    if (!record.data.containsKeyTs("name")) {
                        record.data.put("name", fileInfo.name.get());
                    }

                    MpfsFileMeta meta = fileInfo.getMeta();
                    if (meta.getMediaType().isPresent()) {
                        record.data.put("media_type", meta.getMediaType().get());
                    }

                    if (meta.getUid().isPresent()) {
                        record.data.put("uid", meta.getUid().get());
                    }

                    if (meta.getPreview().isPresent()) {
                        record.data.put("preview", meta.getPreview().get());
                    }

                    if (meta.getShortUrl().isPresent()) {
                        record.data.put("short_url", meta.getShortUrl().get());
                    } else {
                        updateShortUrlIfFileInPublicDirectory(record);
                    }
                }
            } catch (Exception e) {
                logger.error("Can't get mpfs resource info data for anti porno checking, hash=" + hash, e);
                record.addToCheckedFilesCollection = false;
                if (isUserBlockedException(e)) {
                    record.userIsBlocked = true;
                }
            }
        }
    }

    private void updateShortUrlIfFileInPublicDirectory(PornoCheckRecord record) {
        logger.info("File in public directory, updating short url in porno check record.");

        String hash = extractHashForPublicDirectory(record.<String>getTypedData("hash").get());
        try {
            Option<MpfsFileInfo> info = mpfsClient.getFileInfoOByPrivateHash(hash);
            if (info.isPresent()) {
                record.data.put("short_url", info.get().getMeta().getShortUrl().get());
            }
        } catch (Exception e) {
            logger.error("Can't update short url for public folder for anti porno checking, folder hash=" + hash, e);
        }
    }

    private String extractHashForPublicDirectory(String fileHash) {
        return fileHash.substring(0, fileHash.indexOf("=") + 1);
    }

    private boolean isUserBlockedException(Exception e) {
        return Option.ofNullable(e.getMessage())
                .isMatch(s -> s.contains("user blocked"));
    }
}
