package ru.yandex.chemodan.app.stat.antiporno.processors;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.stat.antiporno.PornoCheckRecord;
import ru.yandex.chemodan.app.stat.antiporno.collectors.DataCollector;
import ru.yandex.chemodan.app.stat.antiporno.scores.Score;
import ru.yandex.chemodan.util.http.HttpClientUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadToOutputStreamResponseHandler;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author metal
 */
public class PreviewRecognitionDetector extends AbstractDetector {
    private static final Logger logger = LoggerFactory.getLogger(PreviewRecognitionDetector.class);

    private static final int PORNO_NEURAL_NET_SCORE_NORMALIZER = 255;

    private static final ObjectMapper objectMapper = new ObjectMapper();

    private final DynamicProperty<Double> previewRecognitionDetectorScoreWeight =
            new DynamicProperty<>("antifo.antiporno.detectors.preview.recognition.weight", 1.);

    private final String recognizerUrl;
    private final HttpClient httpClient;

    public PreviewRecognitionDetector(String recognizerUrl, ListF<DataCollector> dataCollectors, HttpClient httpClient) {
        super(dataCollectors);
        this.recognizerUrl = recognizerUrl;
        this.httpClient = httpClient;
    }

    @Override
    protected void detect(PornoCheckRecord record) {
        if (record.data.containsKeyTs("preview")) {
            File2.withNewTempFile(preview -> {
                String defaultPreviewSizeUrl = record.<String>getTypedData("preview").get();
                String previewUrl = UrlUtils.updateParameter(defaultPreviewSizeUrl, "size", "L");
                try {
                    ApacheHttpClientUtils.execute(new HttpGet(previewUrl),
                            httpClient,
                            new ReadToOutputStreamResponseHandler(preview.asAppendOutputStreamTool().getOutput()));

                    HttpPost request = HttpClientUtils.httpPostFile(recognizerUrl, preview);

                    JsonNode result = objectMapper.readTree(ApacheHttpClientUtils.executeReadString(request, true));

                    double scoreValue = result.get("NeuralNetClasses").get("porno").asDouble() / PORNO_NEURAL_NET_SCORE_NORMALIZER;

                    if (scoreValue > 0) {
                        logger.info("Preview recognition algorithm detects porno content in file, score=" + scoreValue);
                        record.scores.add(new Score("preview recognition", scoreValue,
                                previewRecognitionDetectorScoreWeight.get(),
                                "preview recognition algorithm detects porno content"));
                    }
                } catch (Exception e) {
                    logger.error("Can't check with preview recognition detector for porno, url={} :{}", previewUrl, e);
                    record.addToCheckedFilesCollection = false;
                }
            });
        } else {
            record.addToCheckedFilesCollection = false;
        }
    }
}
