package ru.yandex.chemodan.app.stat.limits;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.mongodb.MongoClient;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.app.stat.antiporno.AntipornoContextConfiguration;
import ru.yandex.chemodan.app.stat.antiporno.PornoChecker;
import ru.yandex.chemodan.app.stat.limits.block.BlockDownloadsManager;
import ru.yandex.chemodan.app.stat.limits.block.BlockedDownloadsDao;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.download.DownloadLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.mediatype.MediatypeLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.whitelist.WhitelistRegistry;
import ru.yandex.chemodan.app.stat.worker.LimitMetricUpdaterWorkerService;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.commune.zk2.primitives.group.ZkGroupWatcher;

/**
 * @author Lev Tolmachev
 */
@Configuration
@Import({
        AntipornoContextConfiguration.class
})
public class DiskLimitsContextConfiguration {
    private final DynamicProperty<Integer> blockerThreadCount = DynamicProperty.cons("disk-limit-blocker-executor-thread-count", 8);
    private final DynamicProperty<Integer> nextPeriodBlockerThreadCount = DynamicProperty.cons("disk-limit-next-period-blocker-executor-thread-count", 8);

    @Bean
    public ChannelLimitsRegistry channelLimitsRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        ChannelLimitsRegistry limitsRegistry = new ChannelLimitsRegistry(zkRoot.child("channel_limits"));
        zkManager.addClient(limitsRegistry);
        return limitsRegistry;
    }

    @Bean
    public ChannelLimitsRegistry antiPornoLimitsRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        ChannelLimitsRegistry limitsRegistry = new ChannelLimitsRegistry(zkRoot.child("anti_porno_limits"));
        zkManager.addClient(limitsRegistry);
        return limitsRegistry;
    }

    @Bean
    public LimitedDownloadsDao limitedFilesDao(
            @Qualifier("statMongoClient")
            MongoClient statMongoClient,
            @Value("${mongo.stat.db.name}")
            String dbName,
            @Value("${download-stat.aggregation.period}")
            Duration period)
    {
        return new LimitedDownloadsDao(statMongoClient.getDB(dbName), period);
    }

    @Bean
    public WhitelistRegistry whitelistRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        WhitelistRegistry whitelistRegistry = new WhitelistRegistry(zkRoot.child("whitelist"));
        zkManager.addClient(whitelistRegistry);
        return whitelistRegistry;
    }

    @Bean
    public DownloadLimitsRegistry oneFileLimitsRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        DownloadLimitsRegistry downloadLimitsRegistry = new DownloadLimitsRegistry(zkRoot.child("file-limits"));
        zkManager.addClient(downloadLimitsRegistry);
        return downloadLimitsRegistry;
    }

    @Bean
    public LimitsManager limitsManager(
            @Qualifier("channelLimitsRegistry")
            ChannelLimitsRegistry channelLimitsRegistry,
            @Qualifier("antiPornoLimitsRegistry")
            ChannelLimitsRegistry antiPornoLimitsRegistry,
            LimitedDownloadsDao limitedDownloadsDao,
            WhitelistRegistry whitelistRegistry,
            DownloadLimitsRegistry downloadLimitsRegistry,
            MediatypeLimitsRegistry mediatypeLimitsRegistry,
            PornoChecker pornoChecker)
    {
        return new LimitsManager(channelLimitsRegistry, antiPornoLimitsRegistry, limitedDownloadsDao, whitelistRegistry,
                downloadLimitsRegistry, mediatypeLimitsRegistry, pornoChecker);
    }

    @Bean
    public LimitMetricUpdaterWorkerService limitMetricUpdaterWorkerService(
            LimitsManager limitsManager,
            @Value("${download-stat.aggregation.period}")
            Duration period
    ) {
        return new LimitMetricUpdaterWorkerService(limitsManager, period);
    }

    @Bean
    public MediatypeLimitsRegistry mediatypeLimitsRegistry(
            @Qualifier("zkRoot")
            ZkPath zkRoot,
            ZkManager zkManager)
    {
        ZkPath path = zkRoot.child("mediatype-limits");
        MediatypeLimitsRegistry registry = new MediatypeLimitsRegistry(path);
        zkManager.addClient(new ZkGroupWatcher(path, registry));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public BlockedDownloadsDao blockedFilesDao(
            @Qualifier("statMongoClient")
            MongoClient statMongoClient,
            @Value("${mongo.stat.db.name}")
            String dbName)
    {
        return new BlockedDownloadsDao(statMongoClient.getDB(dbName).getCollection("blocked"));
    }

    @Bean
    public BlockDownloadsManager blockFilesManager(
            LimitedDownloadsDao limitedDownloadsDao,
            BlockedDownloadsDao blockedDownloadsDao,
            @Qualifier("channelLimitsRegistry")
            ChannelLimitsRegistry channelLimitsRegistry,
            @Value("${download-stat.aggregation.period}")
            Duration period,
            @Qualifier("limitBlockerExecutorService")
            ExecutorService limitBlockerExecutorService,
            @Qualifier("limitNextPeriodBlockerExecutorService")
            ExecutorService limitNextPeriodBlockerExecutorService,
            @Value("${download-stat.block.periods.public:-3}")
            int periodsToBlockFilePublic)
    {
        return new BlockDownloadsManager(period, blockedDownloadsDao, limitedDownloadsDao, channelLimitsRegistry,
                limitBlockerExecutorService, limitNextPeriodBlockerExecutorService, periodsToBlockFilePublic);
    }

    @Bean(name = "limitBlockerExecutorService")
    public ExecutorService limitBlockerExecutorService(DynamicPropertyManager dynamicPropertyManager) {
        return createExecutorWithDynamicThreads(dynamicPropertyManager, blockerThreadCount);
    }

    @Bean(name = "limitNextPeriodBlockerExecutorService")
    public ExecutorService limitNextPeriodBlockerExecutorService(DynamicPropertyManager dynamicPropertyManager) {
        return createExecutorWithDynamicThreads(dynamicPropertyManager, nextPeriodBlockerThreadCount);
    }

    private ExecutorService createExecutorWithDynamicThreads(DynamicPropertyManager dynamicPropertyManager, DynamicProperty<Integer> threads) {
        ThreadPoolExecutor executor = new ThreadPoolExecutor(threads.get(), threads.get(),
                1, TimeUnit.MINUTES, new LinkedBlockingQueue<>());

        dynamicPropertyManager.registerWatcher(threads, value -> {
            executor.setMaximumPoolSize(value);
            executor.setCorePoolSize(value);
        });
        return executor;
    }
}
