package ru.yandex.chemodan.app.stat.limits;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.w3c.dom.Node;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.util.DbUtils;
import ru.yandex.chemodan.app.stat.storage.util.DownloadStatIdMarshaller;
import ru.yandex.chemodan.app.stat.storage.util.DownloadStatIdUnmarshaller;
import ru.yandex.commune.mongo.MongoCollection;
import ru.yandex.commune.mongo.MongoDefaultInterceptors;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.commune.mongo.bender.MongoId;
import ru.yandex.commune.mongo.bender.MongoObjectNode;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.FieldLevelUnmarshaller;
import ru.yandex.misc.bender.parse.ParseResult;
import ru.yandex.misc.bender.parse.UnmarshallerContext;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.impl.LruCache;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.clock.Clock;

import static ru.yandex.chemodan.app.stat.storage.DownloadStatId.fromMongoId;

/**
 * @author Lev Tolmachev
 */
public class LimitedDownloadsDao {
    private static final Logger logger = LoggerFactory.getLogger(LimitedDownloadsDao.class);

    private static final BenderConfiguration benderConfiguration = new BenderConfiguration(
            BenderConfiguration.defaultSettings(),
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                    .add(DownloadStatId.class, new DownloadStatIdMarshaller())
                    .add(DownloadStatId.class, new DownloadStatIdUnmarshaller())
                    .add(LimitedInfoWithStatId.class, new LimitedInfoWithStatIdUnmarshaller())
                    .build()
    );

    private final Cache<InstantInterval, MongoCollection<DownloadStatId, LimitedInfoWithStatId>> collectionsCache =
            new LruCache<>(100);

    private final DB db;
    private final Duration period;
    private final Clock clock = Clock.DEFAULT;

    public LimitedDownloadsDao(DB db, Duration period) {
        this.db = db;
        this.period = period;
    }

    public LimitedInfo getLimitedInfo(DownloadStatId statId) {
        return getLimitedInfo(statId, getCurrentCollection());
    }

    public LimitedInfo getLimitedInfo(DownloadStatId statId, InstantInterval interval) {
        return getLimitedInfo(statId, getPeriodCollection(interval));
    }

    public void setBlocked(DownloadStatId statId, boolean auth) {
        setBlocked(statId, auth, getCurrentCollection());
    }

    public void setBlocked(DownloadStatId statId, boolean auth, InstantInterval interval) {
        setBlocked(statId, auth, getPeriodCollection(interval));
    }

    public void setLimited(DownloadStatId statId, String channel, boolean auth, InstantInterval interval) {
        setLimited(statId, channel, auth, getPeriodCollection(interval));
    }

    public void setLimited(DownloadStatId statId, String channel, boolean auth) {
        setLimited(statId, channel, auth, getCurrentCollection());
    }

    public void clearLimited(DownloadStatId statId, String channel, boolean auth) {
        clearLimited(statId, channel, auth, getCurrentCollection());
    }

    public void clearLimited(DownloadStatId statId) {
        getCurrentCollection().remove(DbUtils.getIdQuery(statId));
    }

    private void setLimited(DownloadStatId statId, String channel, boolean auth,
            MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection)
    {
        logger.info("Limit file: {} channel: {}, auth: {}, collection: {}", statId, channel, auth,
                collection.getCollection().getName());

        BasicDBObject set = new BasicDBObject();
        BasicDBObject update = new BasicDBObject("$set", set);

        set.append(channel + "." + (auth ? "a" : "p"), MongoUtils.toMongoValue(clock.now()));

        updateWithRetires(collection, statId, update, true, false);
    }

    private void clearLimited(DownloadStatId statId, String channel, boolean auth,
            MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection)
    {
        logger.info("Unlimit file: {} channel: {}, auth: {}, collection: {}", statId, channel, auth,
                collection.getCollection().getName());

        BasicDBObject set = new BasicDBObject();
        BasicDBObject update = new BasicDBObject("$unset", set);

        set.append(channel + "." + (auth ? "a" : "p"), "");

        updateWithRetires(collection, statId, update);
    }

    private void setBlocked(DownloadStatId statId, boolean auth,
            MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection)
    {
        logger.info("Set file blocked: {} auth: {} collection: {}", statId, auth, collection.getCollection().getName());

        BasicDBObject set = new BasicDBObject();
        BasicDBObject update = new BasicDBObject("$set", set);

        set.append((auth ? "ab" : "pb"), true);

        updateWithRetires(collection, statId, update, true, false);
    }

    private MongoCollection<DownloadStatId, LimitedInfoWithStatId> getCurrentCollection() {
        return getPeriodCollection(clock.now());
    }

    private MongoCollection<DownloadStatId, LimitedInfoWithStatId> getPeriodCollection(Instant now) {
        Instant start = TimeUtils.roundTo(now, period);
        return getPeriodCollection(new InstantInterval(start, start.plus(period)));
    }

    private MongoCollection<DownloadStatId, LimitedInfoWithStatId> getPeriodCollection(final InstantInterval period) {
        return collectionsCache.getFromCacheSome(period,
                () -> new MongoCollection<>(DbUtils.getCollection(db, period, "excess"),
                        LimitedInfoWithStatId.class, benderConfiguration,
                        MongoDefaultInterceptors.defaultInterceptors()));
    }

    public long getLimitedFilesCount(InstantInterval period) {
        return getPeriodCollection(period).findCount(new BasicDBObject());
    }

    public MapF<DownloadStatId, LimitedInfo> getLimitedFiles(InstantInterval period) {
        return getPeriodCollection(period).find(new BasicDBObject()).toMap(
                LimitedInfoWithStatId.getStatIdF(),
                LimitedInfoWithStatId.toLimitedInfoF()
        );
    }

    private LimitedInfo getLimitedInfo(DownloadStatId statId,
            MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection)
    {
        return collection.find(DbUtils.getIdQuery(statId)).firstO().map(LimitedInfoWithStatId.toLimitedInfoF())
                .getOrElse(LimitedInfo.EMPTY);
    }

    private void clearLimited(DownloadStatId statId,
            MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection)
    {
        logger.info("Unlimit file: {}", statId);
        removeWithRetries(collection, statId);
    }

    private void removeWithRetries(final MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection,
            final DownloadStatId statId)
    {
        RetryUtils.retry(3, ((Function0V) () -> collection.remove(DbUtils.getIdQuery(statId))).asFunction0ReturnNull());
    }

    private void updateWithRetires(final MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection,
            final DownloadStatId statId, final BasicDBObject query)
    {
        updateWithRetires(collection, statId, query, false, false);
    }

    private void updateWithRetires(final MongoCollection<DownloadStatId, LimitedInfoWithStatId> collection,
            final DownloadStatId statId, final BasicDBObject query,
            final boolean upsert, final boolean multi)
    {
        RetryUtils.retry(3, () -> collection.update(DbUtils.getIdQuery(statId), query, upsert, multi));
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    private static final class LimitedInfoWithStatId extends LimitedInfo {
        @MongoId
        public final DownloadStatId hid;

        public LimitedInfoWithStatId(MapF<String, Instant> authLimited, MapF<String, Instant> publicLimited,
                boolean authBlocked, boolean publicBlocked, DownloadStatId statId)
        {
            super(authLimited, publicLimited, authBlocked, publicBlocked);
            this.hid = statId;
        }

        public static Function<? super LimitedInfoWithStatId, DownloadStatId> getStatIdF() {
            return limitedInfoWithStatId -> limitedInfoWithStatId.hid;
        }

        public static Function<? super LimitedInfoWithStatId, LimitedInfo> toLimitedInfoF() {
            return withHid -> new LimitedInfo(withHid.authLimited, withHid.publicLimited,
                    withHid.authBlocked, withHid.publicBlocked);
        }
    }

    private static final class LimitedInfoWithStatIdUnmarshaller implements FieldLevelUnmarshaller {
        @Override
        public ParseResult<Object> parseXmlNode(Node node, UnmarshallerContext unmarshallerContext) {
            throw new RuntimeException("xml not supported");
        }

        @Override
        public ParseResult<Object> parseJsonNode(BenderJsonNode json, UnmarshallerContext unmarshallerContext) {
            DownloadStatId id = fromMongoId(((MongoObjectNode) json.getField("hid").get()).getByteArrayValue());

            MapF<String, Instant> publicLimited = Cf.hashMap();
            MapF<String, Instant> authLimited = Cf.hashMap();
            boolean authBlocked = false;
            boolean publicBlocked = false;

            for (String key : json.getFieldNames()) {
                if (!"_id".equals(key)) {
                    MongoObjectNode value = (MongoObjectNode) json.getField(key).get();
                    if (value.isObject() && value.getField("a").isPresent()) {
                        authLimited
                                .put(key, new Instant(((MongoObjectNode) value.getField("a").get()).getDateValue()));
                    }
                    if (value.isObject() && value.getField("p").isPresent()) {
                        publicLimited
                                .put(key, new Instant(((MongoObjectNode) value.getField("p").get()).getDateValue()));
                    }
                    if (key.equals("ab")) {
                        authBlocked = true;
                    }
                    if (key.equals("pb")) {
                        publicBlocked = true;
                    }
                }
            }
            return ParseResult.result(new LimitedInfoWithStatId(
                    authLimited, publicLimited, authBlocked, publicBlocked, id
            ));
        }
    }
}
