package ru.yandex.chemodan.app.stat.limits;

import org.joda.time.Duration;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.stat.TestUtils;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.test.TestHelper;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author Lev Tolmachev
 */
public class LimitedDownloadsDaoTest {

    private DownloadStatId statId;

    private static LimitedDownloadsDao getDao() {
        return new LimitedDownloadsDao(TestUtils.testMongoDb(), Duration.standardDays(1));
    }

    @BeforeClass
    public static void prepareLogger() {
        TestHelper.initialize();
    }

    @Before
    public void before() {
        statId = new DownloadStatId(Random2.R.nextAlnum("6ab7684b26c1cdcf4e556d5237530ec3".length()));
    }

    @Test
    public void limitUnlimit() {

        LimitedDownloadsDao dao = getDao();

        dao.clearLimited(statId);

        Assert.equals(LimitedInfo.EMPTY, dao.getLimitedInfo(statId));

        dao.setLimited(statId, "ch1", true);
        LimitedInfo info = dao.getLimitedInfo(statId);

        Assert.isFalse(info.authBlocked || info.publicBlocked);
        Assert.isEmpty(info.publicLimited.keys());
        Assert.equals(Cf.list("ch1"), info.authLimited.keys());

        dao.setLimited(statId, "ch1", false);
        info = dao.getLimitedInfo(statId);

        Assert.isFalse(info.authBlocked || info.publicBlocked);
        Assert.equals(Cf.list("ch1"), info.publicLimited.keys());
        Assert.equals(Cf.list("ch1"), info.authLimited.keys());

        dao.setLimited(statId, "ch2", false);
        info = dao.getLimitedInfo(statId);

        Assert.isFalse(info.authBlocked || info.publicBlocked);
        Assert.equals(Cf.set("ch1", "ch2"), info.publicLimited.keys().unique());
        Assert.equals(Cf.list("ch1"), info.authLimited.keys());

        dao.clearLimited(statId, "ch1", false);
        info = dao.getLimitedInfo(statId);

        Assert.isFalse(info.authBlocked || info.publicBlocked);
        Assert.equals(Cf.list("ch2"), info.publicLimited.keys());
        Assert.equals(Cf.list("ch1"), info.authLimited.keys());

        dao.clearLimited(statId, "ch1", true);
        info = dao.getLimitedInfo(statId);

        Assert.isFalse(info.authBlocked || info.publicBlocked);
        Assert.equals(Cf.list("ch2"), info.publicLimited.keys());
        Assert.isEmpty(info.authLimited.keys());

        dao.clearLimited(statId, "ch2", false);
        Assert.equals(LimitedInfo.EMPTY, dao.getLimitedInfo(statId));
    }

    @Test
    public void blockedNotLimited() {

        LimitedDownloadsDao dao = getDao();

        dao.clearLimited(statId);
        Assert.equals(LimitedInfo.EMPTY, dao.getLimitedInfo(statId));

        LimitedInfo info = dao.getLimitedInfo(statId);
        Assert.isFalse(info.authBlocked || info.publicBlocked);

        dao.setBlocked(statId, true);
        info = dao.getLimitedInfo(statId);
        Assert.isTrue(info.authBlocked);
        Assert.isFalse(info.publicBlocked);

        dao.setBlocked(statId, false);
        info = dao.getLimitedInfo(statId);
        Assert.isTrue(info.authBlocked);
        Assert.isTrue(info.publicBlocked);
    }

    @Test
    public void blockedLimited() {

        LimitedDownloadsDao dao = getDao();

        dao.clearLimited(statId);
        Assert.equals(LimitedInfo.EMPTY, dao.getLimitedInfo(statId));

        dao.setLimited(statId, "ch1", true);
        dao.setLimited(statId, "ch2", false);
        dao.setLimited(statId, "ch3", true);
        dao.setLimited(statId, "ch7", false);

        LimitedInfo info = dao.getLimitedInfo(statId);
        Assert.isFalse(info.authBlocked || info.publicBlocked);

        dao.setBlocked(statId, true);
        info = dao.getLimitedInfo(statId);
        Assert.isTrue(info.authBlocked);
        Assert.isFalse(info.publicBlocked);

        dao.setBlocked(statId, false);
        info = dao.getLimitedInfo(statId);
        Assert.isTrue(info.authBlocked);
        Assert.isTrue(info.publicBlocked);
    }

}
