package ru.yandex.chemodan.app.stat.limits;

import javax.annotation.Nonnull;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.chemodan.app.stat.antiporno.PornoChecker;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelAndAuth;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelLimits;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.download.DownloadLimits;
import ru.yandex.chemodan.app.stat.limits.download.DownloadLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.mediatype.MediatypeLimits;
import ru.yandex.chemodan.app.stat.limits.mediatype.MediatypeLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.whitelist.WhitelistRegistry;
import ru.yandex.chemodan.app.stat.storage.DownloadStat;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.OneChannelStats;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.RoundRobinCounter;
import ru.yandex.misc.monica.core.blocks.RoundRobinCounterMap;
import ru.yandex.misc.monica.core.blocks.Value;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Lev Tolmachev
 */
public class LimitsManager implements MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(LimitsManager.class);

    private final ChannelLimitsRegistry channelsLimits;
    private final ChannelLimitsRegistry antiPornoLimits;
    private final LimitedDownloadsDao limitedDownloadsDao;
    private final WhitelistRegistry whitelist;
    private final DownloadLimitsRegistry downloadLimitsRegistry;
    private final MediatypeLimitsRegistry mediatypeLimitsRegistry;

    private final PornoChecker pornoChecker;

    @MonicaMetric(description = "Limited files count in last 5 minutes")
    @GroupByDefault
    private final RoundRobinCounter limitedTotal = new RoundRobinCounter(Duration.standardMinutes(5));

    @MonicaMetric(description = "Limited files count in last 5 minutes by channel")
    @GroupByDefault
    private final RoundRobinCounterMap limitedByChannel = new RoundRobinCounterMap(Duration.standardMinutes(5));

    @MonicaMetric(description = "Size of limit table for today", name = "numberOfLimitedToday_max")
    @GroupByDefault
    private final Value<Long> numberOfLimitedToday = new Value<>(0L);

    public LimitsManager(ChannelLimitsRegistry channelsLimits, ChannelLimitsRegistry antiPornoLimits,
                         LimitedDownloadsDao limitedDownloadsDao,
                         WhitelistRegistry whitelistRegistry, DownloadLimitsRegistry downloadLimitsRegistry,
                         MediatypeLimitsRegistry mediatypeLimitsRegistry,
                         PornoChecker pornoChecker) {
        this.channelsLimits = channelsLimits;
        this.antiPornoLimits = antiPornoLimits;
        this.limitedDownloadsDao = limitedDownloadsDao;
        this.whitelist = whitelistRegistry;
        this.downloadLimitsRegistry = downloadLimitsRegistry;
        this.mediatypeLimitsRegistry = mediatypeLimitsRegistry;
        this.pornoChecker = pornoChecker;
    }

    public ChannelLimits changeMediatypeChannelLimits(String mediatype, String channel, boolean auth,
                                                      Option<DataSize> traffic, Option<Long> views) {
        MediatypeLimits currentLimits = mediatypeLimitsRegistry.getLimits(mediatype)
                .getOrElse(MediatypeLimits.consF(mediatype));

        Option<ChannelLimits> currentLimitsO = currentLimits.getLimits(channel);
        ChannelLimits limits = sumLimits(channel, auth, traffic, views, currentLimitsO);
        currentLimits = currentLimits.withChangedChannelLimits(limits);
        if (currentLimits.isEmpty()) {
            mediatypeLimitsRegistry.removeMediatypeLimits(mediatype);
        } else {
            mediatypeLimitsRegistry.setMediatypeLimits(currentLimits);
        }
        return limits;
    }

    public ChannelLimits changeChannelLimits(String channel, boolean auth, Option<DataSize> traffic, Option<Long> views) {
        return changeLimits(channel, auth, traffic, views, channelsLimits);
    }

    public ChannelLimits changeAntiPornoLimits(String channel, boolean auth, Option<DataSize> traffic, Option<Long> views) {
        return changeLimits(channel, auth, traffic, views, antiPornoLimits);
    }

    private ChannelLimits changeLimits(String channel, boolean auth, Option<DataSize> traffic,
                                       Option<Long> views, ChannelLimitsRegistry baseChannelLimits) {
        Option<ChannelLimits> currentLimitsO = baseChannelLimits.getLimits().getO(channel);
        ChannelLimits limits = sumLimits(channel, auth, traffic, views, currentLimitsO);
        if (limits.isEmpty()) {
            baseChannelLimits.removeLimit(limits);
        } else {
            baseChannelLimits.setChannelLimit(limits);
        }
        return limits;
    }

    public ChannelLimits changeDownloadLimits(DownloadStatId id, String channel, boolean auth, Option<DataSize> traffic,
                                              Option<Long> views) {
        Option<DownloadLimits> currentFileLimitsO = downloadLimitsRegistry.getLimits().getO(id);
        Option<ChannelLimits> currentLimitsO = currentFileLimitsO.isPresent()
                ? currentFileLimitsO.get().channelLimits.getO(channel)
                : Option.empty();
        ChannelLimits limits = sumLimits(channel, auth, traffic, views, currentLimitsO);

        downloadLimitsRegistry.setFileLimits(new DownloadLimits(id,
                currentFileLimitsO.isPresent()
                        ? currentFileLimitsO.get().channelLimits.plus1(channel, limits)
                        : Cf.map(channel, limits)
        ));
        return limits;
    }

    private ChannelLimits sumLimits(String channel, boolean auth, Option<DataSize> traffic, Option<Long> views,
                                    Option<ChannelLimits> currentLimitsO) {
        ViewsAndTrafficLimits existingAuthLimits =
                !currentLimitsO.isPresent() ? ViewsAndTrafficLimits.EMPTY : currentLimitsO.get().authLimit;
        ViewsAndTrafficLimits existingPublicLimits =
                !currentLimitsO.isPresent() ? ViewsAndTrafficLimits.EMPTY : currentLimitsO.get().publicLimit;

        return new ChannelLimits(channel,
                auth ? new ViewsAndTrafficLimits(traffic, views) : existingAuthLimits,
                !auth ? new ViewsAndTrafficLimits(traffic, views) : existingPublicLimits);
    }

    private Option<LimitedInfo> getLimitedInfo(Option<LimitedInfo> limitedInfoO, DownloadStatId statId) {
        return limitedInfoO.orElse(Option.of(limitedDownloadsDao.getLimitedInfo(statId)));
    }

    public LimitedInfo getLimitedInfo(DownloadStatId id) {
        return limitedDownloadsDao.getLimitedInfo(id);
    }

    public long getLimitedFilesCount(InstantInterval period) {
        return limitedDownloadsDao.getLimitedFilesCount(period);
    }

    public MapF<DownloadStatId, LimitedInfo> getLimitedFiles(InstantInterval period) {
        return limitedDownloadsDao.getLimitedFiles(period);
    }

    public LimitedInfo getLimitedInfo(DownloadStatId id, InstantInterval interval) {
        return limitedDownloadsDao.getLimitedInfo(id, interval);
    }

    public void updateNumberOfLimitedFilesToday(Duration period) {
        numberOfLimitedToday.set(getLimitedFilesCount(TimeUtils.getCurrentPeriod(period)));
    }

    ListF<ChannelAndAuth> getRestrictions(@Nonnull DownloadStat stats, Option<String> mediatype) {
        MapF<String, ChannelLimits> channelLimits = channelsLimits.getLimits();
        mediatype.ifPresent(
                //override general limits with mediatype limits
                m -> mediatypeLimitsRegistry.getLimits(m)
                        .map(MediatypeLimits::getLimitsByChannel)
                        .ifPresent(channelLimits::putAll));

        // override previous value with limit by id
        downloadLimitsRegistry.getLimits().getO(stats.getId()).ifPresent(
                downloadLimits -> channelLimits.putAll(downloadLimits.channelLimits)
        );

        return calculateRestrictions(stats, channelLimits);
    }

    private ListF<ChannelAndAuth> calculateRestrictions(
            DownloadStat stats, MapF<String, ChannelLimits> channelLimits) {
        if (whitelist.isInWhitelist(stats.getId())) {
            logger.info("Id in whitelist: {}", stats.getId());
            return Cf.list();
        }

        ListF<ChannelAndAuth> result = Cf.arrayList();
        logger.debug("Check limits for stats: {}, limits: {}", stats, channelLimits);


        for (Tuple2<String, OneChannelStats> entry : stats.getChannelStats().entries()) {
            if (!channelLimits.containsKeyTs(entry.get1())) {
                continue;
            }
            OneChannelStats channelStats = entry.get2();

            DataSize traffic = channelStats.getTraffic();
            long views = channelStats.getViews();

            ChannelLimits oneChannelLimits = channelLimits.getTs(entry.get1());
            result.addAll(oneChannelLimits.getLimitedTypes(traffic, views));
        }
        return result;
    }

    public ListF<ChannelAndAuth> checkLimits(DownloadStat stats, Option<String> mediatype) {
        ListF<ChannelAndAuth> restrictions = getRestrictions(stats, mediatype);
        limitAccess(stats.getId(), restrictions);
        return restrictions;
    }

    public void limitAccess(DownloadStatId statId, ListF<ChannelAndAuth> restrictions) {
        Option<LimitedInfo> limitedInfo = Option.empty();
        for (ChannelAndAuth channelAndAuth : restrictions) {
            logger.info("File is limited for download. id: {} channel: {}, auth: {}",
                    statId, channelAndAuth.channel, channelAndAuth.auth);

            limitedInfo = getLimitedInfo(limitedInfo, statId);
            MapF<String, Instant> limits =
                    channelAndAuth.auth ? limitedInfo.get().authLimited : limitedInfo.get().publicLimited;
            if (limits.containsKeyTs(channelAndAuth.channel)) {
                logger.debug("File is already marked as limited");
            } else {
                limitedDownloadsDao.setLimited(statId, channelAndAuth.channel, channelAndAuth.auth);
                limitedTotal.inc();
                limitedByChannel.inc(channelAndAuth.channel);
            }
        }
    }


    public void checkAntiPornoLimits(DownloadStat stats, Option<String> hash, Option<String> name) {
        MapF<String, ChannelLimits> antiPornoLimits = this.antiPornoLimits.getLimits();
        ListF<ChannelAndAuth> limitedTypes = calculateRestrictions(stats, antiPornoLimits);
        if (!limitedTypes.isEmpty()) {
            logger.info("File was caught by anti porno limits. id: {}, hash: {}, name: {}, channel: {}, auth: {}",
                    stats.getId(), hash, name, limitedTypes.get(0).channel, limitedTypes.get(0).auth);
            pornoChecker.checkFileForPorno(stats.getId().toHid(), hash, name);
        }
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "stat-manager",
                new MetricName("stat-manager", "limit"),
                "Limits manager"
        );
    }
}
