package ru.yandex.chemodan.app.stat.limits;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelAndAuth;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelLimits;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.download.DownloadLimits;
import ru.yandex.chemodan.app.stat.limits.download.DownloadLimitsRegistry;
import ru.yandex.chemodan.app.stat.limits.whitelist.WhitelistEntry;
import ru.yandex.chemodan.app.stat.limits.whitelist.WhitelistRegistry;
import ru.yandex.chemodan.app.stat.storage.DownloadStat;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.OneChannelStats;
import ru.yandex.chemodan.app.stat.storage.ViewsAndTraffic;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.test.Assert;

/**
 * @author Lev Tolmachev
 */
public class LimitsManagerTest {

    private LimitsManager limitsManager;

    @Before
    public void init() {
        limitsManager = getLimitsManager();
    }

    @Test
    public void channelLimited() {

        ListF<ChannelAndAuth> limitedTypes = getLimitedTypesForHid(
                new DownloadStatId("test"),
                Cf.map("simple", new OneChannelStats(
                        new ViewsAndTraffic(DataSize.fromBytes(49), 4),
                        new ViewsAndTraffic(DataSize.fromBytes(50), 5))
                ));

        Assert.equals(Cf.<ChannelAndAuth>list(), limitedTypes);

        limitedTypes = getLimitedTypesForHid(
                new DownloadStatId("test"), Cf.map(
                        "simple", new OneChannelStats(
                                new ViewsAndTraffic(DataSize.fromBytes(101), 4),
                                new ViewsAndTraffic(DataSize.fromBytes(101), 5))
                ));

        Assert.equals(Cf.list(new ChannelAndAuth("simple", true)), limitedTypes);

        limitedTypes = getLimitedTypesForHid(new DownloadStatId("test"), Cf.map(
                "simple", new OneChannelStats(
                        new ViewsAndTraffic(DataSize.fromBytes(101), 5),
                        new ViewsAndTraffic(DataSize.fromBytes(101), 5))
        ));

        Assert.equals(Cf.list(new ChannelAndAuth("simple", true), new ChannelAndAuth("simple", false)), limitedTypes);
    }

    @Test
    public void oneFileLimits() {
        ListF<ChannelAndAuth> limitedTypes = getLimitedTypesForHid(new DownloadStatId("limited1"), Cf.map(
                "simple2", new OneChannelStats(
                        new ViewsAndTraffic(DataSize.fromBytes(50), 4),
                        new ViewsAndTraffic(DataSize.fromBytes(50), 5))
        ));

        Assert.equals(Cf.list(new ChannelAndAuth("simple2", false)), limitedTypes);

        limitedTypes = getLimitedTypesForHid(new DownloadStatId("limited1"), Cf.map(
                "simple", new OneChannelStats(
                        new ViewsAndTraffic(DataSize.fromBytes(10), 4),
                        new ViewsAndTraffic(DataSize.fromBytes(10), 5))
        ));

        Assert.equals(Cf.list(new ChannelAndAuth("simple", true)), limitedTypes);
    }

    @Test
    public void whitelist() {
        ListF<ChannelAndAuth> limitedTypes = getLimitedTypesForHid(new DownloadStatId("white1"), Cf.map(
                "simple", new OneChannelStats(
                        new ViewsAndTraffic(DataSize.fromBytes(5000), 400),
                        new ViewsAndTraffic(DataSize.fromBytes(5000), 500))
        ));

        Assert.isEmpty(limitedTypes);
    }

    private static LimitsManager getLimitsManager() {
        return new LimitsManager(new ChannelLimitsRegistry(null) {
            @Override
            public MapF<String, ChannelLimits> getLimits() {
                return Cf.map(
                        "simple", new ChannelLimits("simple",
                                new ViewsAndTrafficLimits(Option.of(DataSize.fromBytes(100)), Option.empty()),
                                new ViewsAndTrafficLimits(Option.of(DataSize.fromBytes(1000)), Option.of(10L))),
                        "simple2", new ChannelLimits("simple2",
                                new ViewsAndTrafficLimits(Option.of(DataSize.fromBytes(1000)), Option.empty()),
                                new ViewsAndTrafficLimits(Option.of(DataSize.fromBytes(100)), Option.of(10L))));
            }
        }, null, null, new WhitelistRegistry(null) {
            @Override
            public SetF<WhitelistEntry> getWhitelist() {
                return Cf.set(
                        new WhitelistEntry(new DownloadStatId("white1"), ""),
                        new WhitelistEntry(new DownloadStatId("white2"), ""));
            }

            @Override
            public boolean isInWhitelist(DownloadStatId statId) {
                return getIds().containsTs(statId);
            }

            @Override
            public SetF<DownloadStatId> getIds() {
                return Cf.set(new DownloadStatId("white1"), new DownloadStatId("white2"));
            }
        }, new DownloadLimitsRegistry(null) {
            @Override
            public MapF<DownloadStatId, DownloadLimits> getLimits() {
                return Cf.map(
                        new DownloadStatId("limited1"),
                        new DownloadLimits(new DownloadStatId("limited1"), Cf.map(
                                "simple", new ChannelLimits("simple",
                                        new ViewsAndTrafficLimits(Option.of(DataSize.fromBytes(10)), Option.empty()),
                                        new ViewsAndTrafficLimits(Option.of(DataSize.fromBytes(100)), Option.of(10L)))))
                );
            }
        }, null, null);
    }

    private ListF<ChannelAndAuth> getLimitedTypesForHid(DownloadStatId id, MapF<String, OneChannelStats> stats) {
        return limitsManager.getRestrictions(new DownloadStat(id, stats), Option.empty());
    }
}
