package ru.yandex.chemodan.app.stat.limits.block;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.stat.DownloadStatChannels;
import ru.yandex.chemodan.app.stat.TimeUtils;
import ru.yandex.chemodan.app.stat.limits.LimitedDownloadsDao;
import ru.yandex.chemodan.app.stat.limits.LimitedInfo;
import ru.yandex.chemodan.app.stat.limits.channel.ChannelLimitsRegistry;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Lev Tolmachev
 */
public class BlockDownloadsManager {
    private static final Logger logger = LoggerFactory.getLogger(BlockDownloadsManager.class);

    private final Duration period;

    private final BlockedDownloadsDao blockedDownloadsDao;
    private final LimitedDownloadsDao limitedDownloadsDao;
    private final ChannelLimitsRegistry channelLimitsRegistry;
    private final ExecutorService limitBlockerExecutorService;
    private final ExecutorService limitNextPeriodBlockerExecutorService;
    private final int periodsToBlockFilePublic;

    private final DynamicProperty<Boolean> parallelBlockerEnable = DynamicProperty.cons("disk-limit-blocker-parallel-version-enabled", true);
    private final DynamicProperty<Boolean> parallelNextPeriodBlockerEnable = DynamicProperty.cons("disk-limit-next-period-blocker-parallel-version-enabled", true);

    public BlockDownloadsManager(Duration period, BlockedDownloadsDao blockedDownloadsDao,
                                 LimitedDownloadsDao limitedDownloadsDao, ChannelLimitsRegistry channelLimitsRegistry,
                                 ExecutorService limitBlockerExecutorService, ExecutorService limitNextPeriodBlockerExecutorService,
                                 int periodsToBlockFilePublic) {
        this.period = period;
        this.blockedDownloadsDao = blockedDownloadsDao;
        this.limitedDownloadsDao = limitedDownloadsDao;
        this.channelLimitsRegistry = channelLimitsRegistry;
        this.limitBlockerExecutorService = limitBlockerExecutorService;
        this.limitNextPeriodBlockerExecutorService = limitNextPeriodBlockerExecutorService;
        this.periodsToBlockFilePublic = periodsToBlockFilePublic;
    }

    public BlockInfo getBlockInfo(DownloadStatId id) {
        return blockedDownloadsDao.getBlockInfo(id);
    }

    public void blockDownload(DownloadStatId id, boolean auth) {
        blockedDownloadsDao.block(id, auth);
        ensureFileLimitedInCurrentPeriod(id, auth);
    }

    public void ensureFileLimitedInCurrentPeriod(DownloadStatId id, boolean auth) {
        ensureFileLimitedInPeriod(id, auth, TimeUtils.getCurrentPeriod(period));
    }

    public void ensureFileLimitedInPeriod(DownloadStatId id, boolean auth, InstantInterval period) {
        ListF<String> channels = channelLimitsRegistry.getLimits().keys();
        LimitedInfo limitedInfo = limitedDownloadsDao.getLimitedInfo(id, period);

        MapF<String, Instant> alreadyLimited = auth ? limitedInfo.authLimited : limitedInfo.publicLimited;
        for (String channel : channels) {
            if (!alreadyLimited.containsKeyTs(channel)) {
                limitedDownloadsDao.setLimited(id, channel, auth, period);
            }
        }

        if (!(auth ? limitedInfo.authBlocked : limitedInfo.publicBlocked)) {
            limitedDownloadsDao.setBlocked(id, auth, period);
        }
    }

    public void blockDownloadsLimitedSeveralTimes() {
        InstantInterval interval = TimeUtils.getCurrentPeriod(period);
        MapF<InstantInterval, MapF<DownloadStatId, LimitedInfo>> limitedFiles = Cf.hashMap();
        for (int i = 0; i < periodsToBlockFilePublic; i++) {
            interval = TimeUtils.getPreviousPeriod(interval);
            limitedFiles.put(interval, limitedDownloadsDao.getLimitedFiles(interval));
        }
        SetF<DownloadStatId> allIds = limitedFiles.values().flatMap(
                MapF::keys
        ).unique();

        ListF<CompletableFuture<Object>> futures = Cf.arrayList();

        for (DownloadStatId id : allIds) {
            ListF<InstantInterval> publicLimitedIntervals = Cf.arrayList();
            for (Tuple2<InstantInterval, MapF<DownloadStatId, LimitedInfo>> t : limitedFiles.entries()) {
                if (t._2.containsKeyTs(id) && t._2.getTs(id).publicLimited.containsKeyTs(DownloadStatChannels.SIMPLE)) {
                    publicLimitedIntervals.add(t._1);
                }
            }

            if (publicLimitedIntervals.size() >= periodsToBlockFilePublic) {
                logger.info("Block file {} in several periods: {}",
                        id, publicLimitedIntervals.map(TimeUtils.intervalToStringF()));

                if (parallelBlockerEnable.get()) {
                    futures.add(sendTaskToExecutor(() -> blockById(id), limitBlockerExecutorService));
                } else {
                    blockById(id);
                }
            }
        }

        try {
            CompletableFutures.allOf(futures).get();
        } catch (Exception e) {
            logger.error("Failed to block content", e);
        }
    }

    private Void blockById(DownloadStatId id) {
        blockedDownloadsDao.block(id, false);
        ensureFileLimitedInCurrentPeriod(id, false);
        return null;
    }

    private ListF<CompletableFuture<Object>> limitBlockedDownloadsInPeriodParallel(InstantInterval interval) {
        logger.info("Parallel limiting blocked files in period: {}", TimeUtils.intervalToString(interval));
        return blockedDownloadsDao.getBlockedDownloads().entries()
                .map(t -> sendTaskToExecutor(() -> limitBlockedDownloadInPeriod(t, interval), limitNextPeriodBlockerExecutorService));
    }

    private void limitBlockedDownloadsInPeriod(InstantInterval interval) {
        logger.info("Limiting blocked files in period: {}", TimeUtils.intervalToString(interval));
        blockedDownloadsDao.getBlockedDownloads().entries().forEach(t -> limitBlockedDownloadInPeriod(t, interval));
    }

    private Void limitBlockedDownloadInPeriod(Tuple2<DownloadStatId, BlockInfo> t, InstantInterval interval) {
        if (t._2.publicBlocked.isPresent()) {
            ensureFileLimitedInPeriod(t._1, false, interval);
        }
        if (t._2.authBlocked.isPresent()) {
            ensureFileLimitedInPeriod(t._1, true, interval);
        }
        return null;
    }

    public void limitBlockedFilesInNextPeriod() {
        InstantInterval nextPeriod = TimeUtils.getNextPeriod(TimeUtils.getCurrentPeriod(period));

        if (parallelNextPeriodBlockerEnable.get()) {
            try {
                CompletableFutures.allOf(limitBlockedDownloadsInPeriodParallel(nextPeriod)).get();
            } catch (Exception e) {
                logger.error("Failed to block limited files in period {}", TimeUtils.intervalToString(nextPeriod), e);
            }
        } else {
            limitBlockedDownloadsInPeriod(nextPeriod);
        }
    }

    public long getBlockedFilesCount() {
        return blockedDownloadsDao.getBlockedDownloadsCount();
    }

    private CompletableFuture<Object> sendTaskToExecutor(Supplier<Object> supplier, ExecutorService executorService) {
        return CompletableFuture.supplyAsync(YandexCloudRequestIdHolder.supplyWithYcrid(supplier), executorService);
    }
}
