package ru.yandex.chemodan.app.stat.limits.block;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import org.joda.time.Instant;
import org.w3c.dom.Node;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.chemodan.app.stat.storage.util.DbUtils;
import ru.yandex.chemodan.app.stat.storage.util.DownloadStatIdMarshaller;
import ru.yandex.chemodan.app.stat.storage.util.DownloadStatIdUnmarshaller;
import ru.yandex.commune.mongo.MongoCollection;
import ru.yandex.commune.mongo.MongoDefaultInterceptors;
import ru.yandex.commune.mongo.MongoUtils;
import ru.yandex.commune.mongo.bender.MongoId;
import ru.yandex.commune.mongo.bender.MongoObjectNode;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.FieldLevelUnmarshaller;
import ru.yandex.misc.bender.parse.ParseResult;
import ru.yandex.misc.bender.parse.UnmarshallerContext;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.clock.Clock;

import static ru.yandex.chemodan.app.stat.storage.DownloadStatId.fromMongoId;

/**
 * @author Lev Tolmachev
 */
public class BlockedDownloadsDao {
    private static final Logger logger = LoggerFactory.getLogger(BlockedDownloadsDao.class);

    private static final BenderConfiguration benderConfiguration = new BenderConfiguration(
            BenderConfiguration.defaultSettings(),
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                    .add(DownloadStatId.class, new DownloadStatIdUnmarshaller())
                    .add(DownloadStatId.class, new DownloadStatIdMarshaller())
                    .add(BlockedInfoWithId.class, new BlockedInfoWithHidUnmarshaller())
                    .build()
    );

    private final MongoCollection<DownloadStatId, BlockedInfoWithId> collection;
    private final Clock clock = Clock.DEFAULT;

    public BlockedDownloadsDao(DBCollection collection) {
        this.collection = new MongoCollection<>(collection, BlockedInfoWithId.class,
                benderConfiguration, MongoDefaultInterceptors.defaultInterceptors());
    }

    public void block(final DownloadStatId id, boolean auth) {
        logger.info("Block file: {} auth: {}", id, auth);
        final BasicDBObject query = new BasicDBObject("$set",
                new BasicDBObject(auth ? "a" : "p", MongoUtils.toMongoValue(clock.now())));
        updateWithRetries(id, query, true, false);
    }

    public void unblock(DownloadStatId id, boolean auth) {
        logger.info("Unblock file: {} auth: {}", id, auth);
        BasicDBObject query = new BasicDBObject("$unset", new BasicDBObject(auth ? "a" : "p", ""));
        updateWithRetries(id, query, false, false);
    }

    private void updateWithRetries(final DownloadStatId id, final BasicDBObject query,
                                   final boolean upsert, final boolean multi) {
        RetryUtils.retry(3, () -> collection.update(DbUtils.getIdQuery(id), query, upsert, multi));
    }

    public void unblockAll(DownloadStatId id) {
        logger.info("Unblock total file: {}", id);
        collection.remove(DbUtils.getIdQuery(id));
    }

    public BlockInfo getBlockInfo(DownloadStatId id) {
        return collection.find(DbUtils.getIdQuery(id)).firstO().map(BlockedInfoWithId.toBlockInfoF())
                .getOrElse(BlockInfo.EMPTY);
    }

    public boolean isBlocked(DownloadStatId id, boolean auth) {
        BlockInfo info = getBlockInfo(id);
        return auth ? info.authBlocked.isPresent() : info.publicBlocked.isPresent();
    }

    public MapF<DownloadStatId, BlockInfo> getBlockedDownloads() {
        return collection.find(new BasicDBObject()).toMap(
                BlockedInfoWithId.getIdF(),
                BlockedInfoWithId.toBlockInfoF()
        );
    }

    public long getBlockedDownloadsCount() {
        return collection.findCount(new BasicDBObject());
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    private static final class BlockedInfoWithId extends BlockInfo {
        @MongoId
        @BenderPart(name="hid")
        private final DownloadStatId id;

        public BlockedInfoWithId(Option<Instant> publicBlocked, Option<Instant> authBlocked,
                                 DownloadStatId id) {
            super(publicBlocked, authBlocked);
            this.id = id;
        }

        public static Function<? super BlockedInfoWithId, BlockInfo> toBlockInfoF() {
            return blockedInfoWithId -> new BlockInfo(
                    blockedInfoWithId.publicBlocked, blockedInfoWithId.authBlocked);
        }

        public static Function<? super BlockedInfoWithId, DownloadStatId> getIdF() {
            return blockedInfoWithId -> blockedInfoWithId.id;
        }
    }

    private static final class BlockedInfoWithHidUnmarshaller implements FieldLevelUnmarshaller {
        @Override
        public ParseResult<Object> parseXmlNode(Node node, UnmarshallerContext unmarshallerContext) {
            throw new RuntimeException("xml not supported");
        }

        @Override
        public ParseResult<Object> parseJsonNode(BenderJsonNode json, UnmarshallerContext unmarshallerContext) {
            DownloadStatId id = fromMongoId(((MongoObjectNode) json.getField("hid").get()).getByteArrayValue());

            Option<Instant> authBlocked = getInstant(json, "a");
            Option<Instant> publicBlocked = getInstant(json, "p");

            return ParseResult.result(new BlockedInfoWithId(publicBlocked, authBlocked, id));
        }

        private Option<Instant> getInstant(BenderJsonNode json, String field) {
            return json.getField(field).map(benderJsonNode -> new Instant(((MongoObjectNode) benderJsonNode).getDateValue()));
        }
    }
}
