package ru.yandex.chemodan.app.stat.limits.download;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.stat.storage.DownloadStatId;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.Zk;
import ru.yandex.commune.zk2.primitives.ZkPathClient;
import ru.yandex.commune.zk2.primitives.group.GroupWatcherListener;
import ru.yandex.commune.zk2.primitives.group.ZkGroupWatcher;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Lev Tolmachev
 */
public class DownloadLimitsRegistry extends ZkPathClient implements GroupWatcherListener {
    private static final Logger logger = LoggerFactory.getLogger(DownloadLimitsRegistry.class);

    private final ZkGroupWatcher groupWatcher;
    private MapF<DownloadStatId, DownloadLimits> limits = Cf.map();

    public DownloadLimitsRegistry(ZkPath zkPath) {
        super(zkPath);
        groupWatcher = new ZkGroupWatcher(zkPath, this);
    }

    public MapF<DownloadStatId, DownloadLimits> getLimits() {
        return limits;
    }

    public void setFileLimits(DownloadLimits downloadLimits) {
        logger.info("Save file limits: {}", downloadLimits);
        saveLimits(limits.plus1(downloadLimits.getId(), downloadLimits));
    }

    public void clearFileLimits(DownloadStatId statId) {
        logger.info("Clear file limits: {}", statId);
        saveLimits(limits.filterKeys(Cf.Object.equalsF(statId).notF()));
    }

    private void saveLimits(MapF<DownloadStatId, DownloadLimits> limitsMap) {
        int i = 1;
        for (ListF<DownloadLimits> page : limitsMap.values().toList().paginate(100)) {
            ZkPath listPath = path.child("list-" + i);
            zk().deleteIfExists(listPath);
            zk().createOrUpdate(listPath, DownloadLimits.PS.getSerializer().serializeListJson(page));
            i++;
        }
        for (ZkPath child : zk().getChildren(path)) {
            if (child.getName().startsWith("list-")) {
                String tail = StringUtils.substringAfter(child.getName(), "-");
                Option<Long> number = Cf.Long.parseSafe(tail);
                if (number.isPresent() && number.get() >= i) {
                    zk().delete(child);
                }
            }
        }
    }

    @Override
    public void initialize(Zk zk) {
        super.initialize(zk);
        groupWatcher.initialize(zk);
    }

    @Override
    public void expired() {
        super.expired();
        groupWatcher.expired();
    }

    @Override
    public void groupChanged(ListF<byte[]> members) {
        limits = members.flatMap(bytes -> DownloadLimits.PS.getParser().parseListJson(new ByteArrayInputStreamSource(bytes)))
                .toMapMappingToKey(DownloadLimits.getIdFieldF());
    }
}
